/*******************************************************************************
 * Copyright (c) 2022 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.ui;

import java.awt.Dimension;
import java.awt.Frame;
import java.util.function.Supplier;

import org.eclipse.swt.awt.SWT_AWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.internal.DPIUtil;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.simantics.utils.threads.AWTThread;
import org.simantics.utils.threads.ThreadUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This workaround-listener is intended only for fixing an age-old problem with
 * {@link SWT_AWT#new_Frame(Composite)} where the constructed Frame does not
 * automatically get resized from zero height to the containing SWT composite's
 * height during initial layout and rendering causing the UI to require manual
 * resizing from the user to start showing graphics in the AWT Frame.
 * 
 * @author Tuukka Lehtonen
 * @since 1.52.0
 */
@SuppressWarnings("restriction")
public class InitialFrameSizeFixListener extends ControlAdapter {

	private static final Logger LOGGER = LoggerFactory.getLogger(InitialFrameSizeFixListener.class);

	private Control control;
	private Supplier<Frame> frameProvider;

	public InitialFrameSizeFixListener(Control listenedControl, Supplier<Frame> frameProvider) {
		this.control = listenedControl;
		this.frameProvider = frameProvider;
		control.addControlListener(this);
	}

	private static boolean needToFixDimensions(Point swtSize, Dimension awtSize) {
		Point p = swtSize;
		Dimension d = awtSize;
		return p.x > 0 && d.width == 0 || p.y > 0 && d.height == 0;
	}

	public void controlResized(ControlEvent e) {
		assert e != null;
		assert Display.getCurrent() != null; // On SWT event thread

		if (frameProvider == null)
			return;

		Frame f = frameProvider.get();
		if (f == null)
			return;

		Composite c = (Composite) e.widget;
		Point dsp = c.getSize();
		if (dsp.x == 0 || dsp.y == 0)
			return;

		// Scale up to internal SWT coordinates to make them comparable to AWT coordinates
		Point p = DPIUtil.autoScaleUp(dsp);

		Dimension d = f.getSize();
		LOGGER.trace("{}: PHASE 1: {} vs. {}", this, p, d);
		if (needToFixDimensions(p, d)) {
			ThreadUtils.asyncExec(AWTThread.getThreadAccess(), () -> {
				Dimension d2 = f.getSize();
				LOGGER.trace("{}: PHASE 2: {} vs. {}", InitialFrameSizeFixListener.this, p, d2);
				if (needToFixDimensions(p, d2)) {
					LOGGER.trace("{}: PHASE 3: forcefully fix AWT Frame size to {} from {}", InitialFrameSizeFixListener.this, p, d2);
					f.setSize(p.x, p.y);
					SWTUtils.asyncExec(control, InitialFrameSizeFixListener.this::removeListener);
				}
			});
		} else {
			removeListener();
		}
	}

	private void removeListener() {
		if (!control.isDisposed()) {
			control.removeControlListener(this);
			LOGGER.trace("{}: REMOVE LISTENER", this);
		}
	}

}