/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.graphviz.internal.xdot;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;

import org.simantics.graphviz.Edge;
import org.simantics.graphviz.Graph;
import org.simantics.graphviz.IGraph;
import org.simantics.graphviz.IGraphPart;

public class DrawCommandParser {

	public static double readNumber(Reader s) throws IOException {
		StringBuilder b = new StringBuilder();
		while( true ) {
			int c = s.read();
			if(c == ' ' || c <= 0)
				break;
			b.append((char)c);
		}
		return Double.parseDouble(b.toString());
	}
	
	public static int readInteger(Reader s) throws IOException {
		StringBuilder b = new StringBuilder();
		while( true ) {
			int c = s.read();
			if(c == ' ' || c <= 0)
				break;
			b.append((char)c);
		}
		return Integer.parseInt(b.toString());
	}
	
	public static double[] readPoints(Reader s) throws IOException {
		int n = readInteger(s);
		double[] points = new double[n*2];
		for(int i=0;i<n*2;++i)
			points[i] = readNumber(s);
		return points;
	}
	
	public static String readString(Reader s) throws IOException {
		int n = readInteger(s);
		s.read();
		char[] chars = new char[n];
		for(int i=0;i<n;++i) {
			char c = (char)s.read();
			if(c == '\\')
				c = (char)s.read();
			chars[i] = c;
		}
		s.read();
		return new String(chars);
	}
	
	public static void parse(Collection<DrawCommand> commands, Reader s) throws IOException {
	    boolean hadSetStyle = false;
		while(true) {
			int commandChar = s.read();
			s.read();
			if(commandChar <= 0) {
			    if(hadSetStyle)
			        commands.add(new SetStyle("solid"));
				return;
			}
			switch(commandChar) {
			case 'E': {
				double x = readNumber(s);
				double y = readNumber(s);
				double w = readNumber(s);
				double h = readNumber(s);
				commands.add(new FilledShape(Shapes.createEllipse(x, y, w, h)));
				break;
			}
			case 'e': {
				double x = readNumber(s);
				double y = readNumber(s);
				double w = readNumber(s);
				double h = readNumber(s);
				commands.add(new UnfilledShape(Shapes.createEllipse(x, y, w, h)));
				break;
			}
			case 'P': {
				commands.add(new FilledShape(Shapes.createPolygon(readPoints(s))));
				break;
			}
			case 'p':
				commands.add(new UnfilledShape(Shapes.createPolygon(readPoints(s))));
				break;
			case 'L':
				commands.add(new UnfilledShape(Shapes.createPolyline(readPoints(s))));
				break;
			case 'B':				
				commands.add(new UnfilledShape(Shapes.createCubicSegments(readPoints(s))));
				break;
			case 'b':
				commands.add(new FilledShape(Shapes.createCubicSegments(readPoints(s))));
				break;
			case 'T': {
				double x = readNumber(s);
				double y = readNumber(s);
				int j = readInteger(s);
				double w = readNumber(s);
				String text = readString(s);
				commands.add(new Text(x, y, j, w, text));
				break;
			}
			case 'C':
				commands.add(new SetFillColor(readString(s)));
				break;
			case 'c':
				commands.add(new SetPenColor(readString(s)));
				break;
			case 'F': {
				double points = readNumber(s);
				String font = readString(s);
				commands.add(new SetFont(points, font));
				break;
			}
			case 'S':
			    hadSetStyle = true;
				commands.add(new SetStyle(readString(s)));
				break;
			case 'I': {
				double x = readNumber(s);
				double y = readNumber(s);
				double w = readNumber(s);
				double h = readNumber(s);
				String file = readString(s);
				commands.add(new DrawImage(x, y, w, h, file));
				break;
			}
			}
		}
	}
	
	public static void parse(Collection<DrawCommand> commands, String s) {
		if(s != null)
			try {
				s = s.replace("\\\r\n", "");
				parse(commands, new StringReader(s));
			} catch(IOException e) {
				e.printStackTrace();
			}
	}

	private static void parse(Collection<DrawCommand> commands, IGraph graph) {
	    for(IGraphPart part : graph.getParts()) {
            parse(commands, part.get("_draw_"));
            parse(commands, part.get("_ldraw_"));
            if(part instanceof Edge) {
                parse(commands, part.get("_hdraw_"));
                parse(commands, part.get("_tdraw_"));
                parse(commands, part.get("_hldraw_"));
                parse(commands, part.get("_tldraw_"));
            }
            else if(part instanceof IGraph) {
                parse(commands, (IGraph)part);
            }
        }
	}
	
	private static void parse(Collection<DrawCommand> commands, Map<IGraphPart,Collection<DrawCommand>> partCommands, IGraph graph) {
	    for(IGraphPart part : graph.getParts()) {
	    	Collection<DrawCommand> pCommands = new ArrayList<DrawCommand>();
	    	parse(pCommands, part.get("_draw_"));
            parse(pCommands, part.get("_ldraw_"));
            if(part instanceof Edge) {
                parse(pCommands, part.get("_hdraw_"));
                parse(pCommands, part.get("_tdraw_"));
                parse(pCommands, part.get("_hldraw_"));
                parse(pCommands, part.get("_tldraw_"));
            } else if(part instanceof IGraph) {
                parse(commands, partCommands,(IGraph)part);
            }
            partCommands.put(part, pCommands);
            commands.addAll(pCommands);
            
        }
	}
	
	public static DrawCommand[] parse(Graph graph) {
		Collection<DrawCommand> commands = new ArrayList<DrawCommand>();
		parse(commands, graph);
		return commands.toArray(new DrawCommand[commands.size()]);
	}
	
	public static void parse(Graph graph, Collection<DrawCommand> commands, Map<IGraphPart,Collection<DrawCommand>> partCommands) {
		parse(commands,partCommands, graph);
		
	}
	
}
