/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.graphviz;

import java.io.PrintStream;
import java.util.HashSet;
import java.util.Set;

/**
 * An edge of a graph.
 * 
 * @author Hannu Niemist
 */
public class Edge extends AbstractGraphPart {

	Identifiable tail;
	Identifiable head;
	
	public Edge(IGraph graph, Identifiable tail, Identifiable head) {
        super(graph);
        this.tail = tail;
        this.head = head;
    }
	
	private static IGraph chooseParentGraph(IGraph a, IGraph b) {
	    if(a == b)
	        return a;
	    Set<IGraph> graphs = new HashSet<IGraph>();
	    while(true) {
	        graphs.add(a);
	        if(a instanceof Identifiable)
	            a = ((Identifiable)a).getParent();
	        else
	            break;
	    }
	    while(true) {
	        if(graphs.contains(b))
	            return b;
	        if(b instanceof Identifiable)
                b = ((Identifiable)b).getParent();
            else
                break;
	    }
	    throw new IllegalArgumentException("Tried to connect nodes that do not belong to the same graph.");
	}
	
	public Edge(Identifiable tail, Identifiable head) {
        this(chooseParentGraph(tail.getParent(), head.getParent()), tail, head);
    }
	
    /**
     * Text label attached to objects. 
     */
	public void setLabel(String label) {
		set("label", label);
	}
	
	/**
	 * Text label to be placed near head of edge.
	 */
	public void setHeadLabel(String label) {
	    set("headlabel", label);
	}
	
	
	/**
	 * Text label to be placed near tail of edge.
	 */
	public void setTailLabel(String label) {
	    set("taillabel", label);
    }
	
	/**
	 * <p>Set edge type for drawing arrowheads. This indicates which ends of the edge should be decorated with an arrowhead. The actual style of the arrowhead can be specified using the arrowhead and arrowtail attributes.</p>
	 * <p>Alternatives: <tt>forward back both none</tt></p> 
	 */
    public void setDir(String label) {
        set("dir", label);
    }
    
    /**
     * Set style for node or edge. For cluster subgraph, if "filled", the cluster box's background is filled.
     */
    public void setStyle(String style) {
        set("style", style);
    }
    
    /**
     * If false, the edge is not used in ranking the nodes.
     */
    public void setConstraint(boolean constraint) {
        set("constraint", Boolean.toString(constraint));
    }
    
    /**
     * <p>
     * Basic drawing color for graphics, not text. For the latter, use the
     * fontcolor attribute. For edges, the value can either be a single color or
     * a colorList. In the latter case, the edge is drawn using parallel splines
     * or lines, one for each color in the list, in the order given. The head
     * arrow, if any, is drawn using the first color in the list, and the tail
     * arrow, if any, the second color. This supports the common case of drawing
     * opposing edges, but using parallel splines instead of separately routed
     * multiedges.
     * </p>
     * <p>Supports arbitrary RGB(A) colors in format #xxxxxx, where
     * each x is a hex digit. Supports also color names in x11 color chart.</p>
     */
    public void setColor(String color) {
        set("color", color);
    }
    
    /**
     * Color used for text.
     */
    public void setFontColor(String color) {
        set("fontcolor", color);
    }
	
	/**
	 * <p>Sets the shape of the arrow head</p>
	 * 
	 * <p>Arrow shapes can be specified and named using the following simple grammar. Literal characters are given in single quotes. Square brackets [ and ] enclose optional items. Vertical bars | separate alternatives.
	 * </p>
	 * <pre>
	 * arrowname   :   aname [ aname [ aname [ aname ] ] ]
	 * aname   :   [ modifiers ] shape
	 * modifiers   :   [ 'o' ] [ side ]
	 * side    :   'l'
	 *         |   'r'
	 * shape   :   box
	 *         |   crow
	 *         |   diamond
	 *         |   dot
	 *         |   inv
	 *         |   none
	 *         |   normal
	 *         |   tee
	 *         |   vee
	 * </pre>
	 * As for the modifiers:
	 * <ul>
	 * <li> 'l' Clip the shape, leaving only the part to the left of the edge.</li>
	 * <li> 'r' Clip the shape, leaving only the part to the right of the edge.</li>
	 * <li> 'o' Use an open (non-filled) version of the shape.</li>
	 * </ul>
	 * <p>Left and right are defined as those directions determined by looking from the edge towards the point where the arrow "touches" the node.
	 * </p>
	 * <p>Note that the first arrow shape specified occurs closest to the node. Subsequent arrow shapes, if specified, occur further from the node.
	 * </p>
	 */
	public void setArrowhead(String arrowName) {
	    set("arrowhead", arrowName);
	}
	
	/**
	 * <p>Sets the shape of the arrow tail</p>
	 * 
	 * @see Edge#setArrowhead
	 */
	public void setArrowtail(String arrowName) {
        set("arrowtail", arrowName);
    }
	
	public void setArrowsize(double arrowSize) {
        set("arrowsize", Double.toString(arrowSize));
    }
	
	public Identifiable getTail() {
		return tail;
	}

	public Identifiable getHead() {
		return head;
	}

	@Override
	public void write(PrintStream s) {
		s.print(tail.getId());
		s.print(" -> ");
		s.print(head.getId());
		writeAttributes(s);
	}

}
