/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.charts.query;

import java.util.Arrays;

import org.simantics.charts.ontology.ChartResource;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.binding.Binding;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.subscription.SubscriptionsQuery;
import org.simantics.modeling.subscription.SubscriptionsQuery.SubscriptionsResult;
import org.simantics.trend.configuration.TrendItem;
import org.simantics.trend.configuration.TrendItem.Renderer;

/**
 * This query reads chart and item data. 
 *  
 * @author toni.kalajainen
 */
public class ChartAndSubscriptionItemReadQuery extends ResourceRead<ChartAndSubscriptionItemData> {

	public ChartAndSubscriptionItemReadQuery(Resource chartItem) {
		super(chartItem);
	}

	@Override
	public ChartAndSubscriptionItemData perform(ReadGraph g) throws DatabaseException {
		ChartAndSubscriptionItemData data = new ChartAndSubscriptionItemData();

    	Layer0 L0 = Layer0.getInstance(g);
    	ModelingResources MOD = ModelingResources.getInstance(g);
    	ChartResource CHART = ChartResource.getInstance(g);
    	Resource chartItem = resource;
        Resource chart = g.getPossibleObject(chartItem, L0.PartOf);
        Resource model = g.syncRequest(new PossibleModel( chart ));
        Resource subscriptionItem = g.getPossibleObject(chartItem, CHART.Chart_Item_HasSubscriptionItem);
        Resource subscription = subscriptionItem==null ? null : g.getPossibleObject(subscriptionItem, L0.PartOf);
		SubscriptionsResult subscriptions = g.sync( new SubscriptionsQuery(model) );

		TrendItem trendItem = g.sync( new TrendItemQuery(chartItem) );

		if ( subscriptionItem != null ) {
			Variable configuration = Variables.getConfigurationContext(g, subscription);
			Binding rviBinding = g.getService(Databoard.class).getBindingUnchecked( RVI.class );
			RVI rvi = g.getRelatedValue(subscriptionItem, MOD.Subscription_Item_VariableId, rviBinding);
			data.variableReference = rvi.toPossibleString(g, configuration);
		}

		data.index = trendItem.index;
    	data.chartName = g.getPossibleRelatedValue(chart, L0.HasName, Bindings.STRING);
		data.subscriptions = subscriptions.toNames();		
    	data.subscriptions = subscriptions.names.toArray( new String[subscriptions.names.size()]);
    	data.drawmode = trendItem.drawMode;
    	data.binaryMode = trendItem.renderer == Renderer.Binary;
    	data.subscription = subscriptions.getName(subscription);
    	data.scale = trendItem.scale;
    	if ( data.subscription==null ) data.subscription="";

    	data.min = g.getPossibleRelatedValue(chartItem, CHART.Chart_Item_ScaleMode_Min, Bindings.DOUBLE);  
    	data.max = g.getPossibleRelatedValue(chartItem, CHART.Chart_Item_ScaleMode_Max, Bindings.DOUBLE);

    	data.strokeWidth = trendItem.customStrokeWidth;
    	data.color = trendItem.customColor != null ? Arrays.copyOf(trendItem.customColor, trendItem.customColor.length) : null;

    	data.hasSubscriptionItem = subscriptionItem != null; 
    	if ( data.hasSubscriptionItem ) {
        	data.label = g.getPossibleRelatedValue(subscriptionItem, L0.HasLabel, Bindings.STRING);  
    		data.bias = g.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_Bias, Bindings.DOUBLE);
    		data.gain = g.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_Gain, Bindings.DOUBLE);
    		data.deadband = g.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_Deadband, Bindings.DOUBLE);
    		data.interval = g.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_SamplingInterval, Bindings.DOUBLE);
    		data.unit = g.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_Unit, Bindings.STRING);
    	} else {
    		data.variableReference = trendItem.label;
    	}

		return data;
	}

}
