/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;

import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.PrimitiveQueryProcessor;
import org.simantics.browsing.ui.NodeContext.ConstantKey;
import org.simantics.browsing.ui.common.processors.IsExpandedProcessor;

/**
 * General utilities for dealing with {@link NodeContext} instances.
 * 
 * @author Tuukka Lehtonen
 */
public final class NodeContextUtil {

    public interface NodeContextFactory {
        NodeContext create(Object input);
    }

    /**
     * @param <T>
     * @param clazz
     * @return input of the specified class
     * @throws ClassCastException if the input class does not match the
     *         specified class
     * @throws NullPointerException if the input is null
     */
    @SuppressWarnings("unchecked")
    public static <T> T getInput(NodeContext context, Class<T> clazz) throws ClassCastException {
        Object o = context.getConstant(BuiltinKeys.INPUT);
        if (o == null)
            throw new NullPointerException("null input");
        //return clazz.cast(o);
        return (T) o;
    }

    public static <T> NodeContext withConstant(NodeContext context, ConstantKey<T> key, T value) {
        Set<ConstantKey<?>> keys = context.getKeys();
        Object[] keyValuePairs = new Object[2 + 2 * keys.size()];
        int index = 0;
        keyValuePairs[index++] = key;
        keyValuePairs[index++] = value;
        for(ConstantKey<?> k : keys) {
            keyValuePairs[index++] = k;
            keyValuePairs[index++] = context.getConstant(k);
        }
        return NodeContextBuilder.buildWithData(keyValuePairs);
    }

    /**
     * @param <T>
     * @param clazz
     * @return <code>null</code> if input is <code>null</code> or if the class does not match
     */
    @SuppressWarnings("unchecked")
    public static <T> T tryGetInput(NodeContext context, Class<T> clazz) {
        Object o = context.getConstant(BuiltinKeys.INPUT);
        if (o != null && clazz.isInstance(o))
            //return clazz.cast(o);
            return (T) o;
        return null;
    }

    /**
     * A utility method for transforming a collection of objects into an
     * INodeContext array which is the return value of a ViewpointFactory.
     * 
     * <p>
     * The INodeContext's are constructed using
     * {@link NodeContextBuilder#buildWithInput(Object)}.
     * </p>
     * 
     * @param children
     * @return the specified children wrapped into simple <code>INodeContext</code>s
     */
    public static NodeContext[] toContextsWithInput(Collection<?> children) {
        NodeContext[] resultContexts = new NodeContext[children.size()];
        int index = 0;
        for (Object child : children) {
            if(child == null) throw new NullPointerException("Null data.");
            resultContexts[index++] = NodeContextBuilder.buildWithInput(child);
        }
        return resultContexts;
    }

    public static Collection<NodeContext> toContextCollectionWithInput(Object source, Collection<?> children) {
        if(children == null) throw new NullPointerException("Null data (from " + source.getClass().getName() + ").");
        ArrayList<NodeContext> resultContexts = new ArrayList<NodeContext>(children.size());
        for (Object child : children) {
            if(child == null) throw new NullPointerException("Null data (from " + source.getClass().getName() + ").");
            resultContexts.add(NodeContextBuilder.buildWithInput(child));
        }
        return resultContexts;
    }

    public static Collection<NodeContext> toContextCollectionWithInput(Object ... children) {
        ArrayList<NodeContext> resultContexts = new ArrayList<NodeContext>(children.length);
        for (Object child : children) {
            if(child == null) throw new NullPointerException("Null data.");
            resultContexts.add(NodeContextBuilder.buildWithInput(child));
        }
        return resultContexts;
    }

    public static NodeContext[] toContextsWithInput(Object ... inputs) {
        NodeContext[] resultContexts = new NodeContext[inputs.length];
        int index = 0;
        for (Object child : inputs) {
            if(child == null) throw new NullPointerException("Null data.");
            resultContexts[index++] = NodeContextBuilder.buildWithInput(child);
        }
        return resultContexts;
    }

    /**
     * A utility method for transforming a collection of objects into an
     * INodeContext array which is the return value of a ViewpointFactory.
     * 
     * <p>
     * The INodeContext's are constructed using the specified factory.
     * </p>
     * 
     * @param children
     * @return the specified children wrapped into INodeContext's through the
     *         specified <code>factory</code>
     */
    public static NodeContext[] toContexts(Collection<?> children, NodeContextUtil.NodeContextFactory factory) {
        NodeContext[] resultContexts = new NodeContext[children.size()];
        int index = 0;
        for (Object child : children) {
            if(child == null) throw new NullPointerException("Null data.");
            resultContexts[index++] = factory.create(child);
        }
        return resultContexts;
    }

    public static void collapseAllNodes(GraphExplorer explorer) {
        PrimitiveQueryProcessor<?> pqp =  explorer.getPrimitiveProcessor(BuiltinKeys.IS_EXPANDED);
        if (pqp instanceof IsExpandedProcessor) {
            IsExpandedProcessor iep = (IsExpandedProcessor) pqp;
            iep.getExpanded().forEach(n -> iep.replaceExpanded(n, false));
        }
    }

    public static void collapseNodesUnder(GraphExplorer explorer, Set<NodeContext> collapsedParents) {
        PrimitiveQueryProcessor<?> pqp =  explorer.getPrimitiveProcessor(BuiltinKeys.IS_EXPANDED);
        if (pqp instanceof IsExpandedProcessor) {
            IsExpandedProcessor iep = (IsExpandedProcessor) pqp;
            iep.getExpanded().stream()
            .filter(n -> isParent(explorer, collapsedParents, n))
            .forEach(n -> iep.replaceExpanded(n, false));
        }
    }

    private static boolean isParent(GraphExplorer explorer, Set<NodeContext> testedParents, NodeContext testedNode) {
        if (testedNode == null)
            return false;
        if (testedParents.contains(testedNode))
            return true;
        return isParent(explorer, testedParents, explorer.getParentContext(testedNode));
    }

}
