package org.simantics.browsing.ui.swt;

import java.io.IOException;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.processors.FilterSelectionRequestQueryProcessor;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.layer0.Layer0;
import org.simantics.utils.ui.SWTUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Resource Type specific filters. 
 * 
 * Stores the filter string per type basis, and the loads the stored filter when object with the same type is encountered.
 * 
 * @author Marko Luukkainen <marko.luukkainen@semantum.fi>
 *
 */
public class TypeContextFilterArea extends RootFilterArea implements Widget {

	private static final Logger LOGGER = LoggerFactory.getLogger(TypeContextFilterArea.class);

	String preferencePrefix;

 	String typeUri;

 	/**
 	 * Creates new Filter Area.
 	 * @param explorer
 	 * @param queryProcessor
 	 * @param support
 	 * @param parent
 	 * @param id unique id of the UI widget. The id is used to generate preference store keys to store filters.
 	 * @param style
 	 */
	public TypeContextFilterArea(GraphExplorer explorer, FilterSelectionRequestQueryProcessor queryProcessor, WidgetSupport support,
			Composite parent, String id, int style) {
		super(explorer, queryProcessor, parent, style);
		this.preferencePrefix = id +"_TypeFilter_"; //$NON-NLS-1$
		support.register(this);
		
	}
	
	@Override
	public void setInput(ISessionContext context, Object input) {
		final Resource resource = org.simantics.utils.ui.AdaptionUtils.adaptToSingle(input, Resource.class);
		if (resource == null)
			return;
		context.getSession().asyncRequest(new ReadRequest() {
			@Override
			public void run(ReadGraph graph) throws DatabaseException {
				setInput(graph, resource);
			}
		});
	}
	
	public void setInput(ReadGraph graph, Resource resource) throws DatabaseException{
		Resource type = getType(graph, resource);
		String typeUri = type == null ? null : graph.getPossibleURI(type);
		setTypeURI(typeUri);
	}

	public void setTypeURI(String typeUri) {
		if (typeUri != null)
			SWTUtils.asyncExec(Display.getDefault(), () -> load(typeUri));
	}

	protected Resource getType(ReadGraph graph, Resource resource) throws DatabaseException {
		return graph.getPossibleType(resource, Layer0.getInstance(graph).Entity);
	}

	@Override
	protected synchronized void applyFilter(NodeContext context, String filter, boolean updateUI) {
		super.applyFilter(context, filter,updateUI);
		if (typeUri != null)
			store(typeUri, filter);
	}
	
	private synchronized void store(String typeUri, String filter) {
		ScopedPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);
		if (filter != null && filter.length() > 0)
			store.setValue(preferencePrefix+typeUri, filter);
		else
			store.setValue(preferencePrefix+typeUri, ""); //$NON-NLS-1$
		try {
			store.save();
		} catch (IOException e) {
			LOGGER.error("Failed to save filter preference '{}' for type-contextual filter area with type URI {}", //$NON-NLS-1$
					filter, typeUri, e);
		}
	}
	
	private synchronized void load(String typeUri) {
		this.typeUri = typeUri;
		ScopedPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);
		String filter = store.getString(preferencePrefix+typeUri);
		if (filter != null && filter.length() > 0) {
			applyFilter(filter);
		}
	}

}
