/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.swt.stubs.BrowsingResource;
import org.simantics.browsing.ui.swt.widgets.Button;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.request.Read;

public class ModelledButton implements ModelledControl {

	final private Resource configuration;
	
	public ModelledButton(Resource configuration) {
		this.configuration = configuration;
	}
	
	@Override
	public Control create(Composite parent, final IWorkbenchSite site, final ISessionContext context, final WidgetSupport support) throws DatabaseException {

        LocalResourceManager resourceManager = new LocalResourceManager(JFaceResources.getResources(), parent);
		
		Button button = new Button(parent, support, SWT.READ_ONLY);
		
        GridData gridData = Simantics.getSession().syncRequest(new Read<GridData>() {

            @Override
            public GridData perform(ReadGraph graph) throws DatabaseException {
                BrowsingResource br = BrowsingResource.getInstance(graph);
                GridData data = new GridData(SWT.LEFT, SWT.CENTER, false, false);                
                Integer span = graph.getPossibleRelatedValue(configuration, br.Control_HorizontalSpan, Bindings.INTEGER);
                if(span != null) data.horizontalSpan = span;
                Boolean grabExcessHorizontalSpace = graph.getPossibleRelatedValue(configuration, br.Control_HorizontalGrab, Bindings.BOOLEAN);
                if(grabExcessHorizontalSpace != null) data.grabExcessHorizontalSpace = grabExcessHorizontalSpace;
                return data;
            }

        });
        
        button.setLayoutData(gridData);
		
		String text = Simantics.getSession().syncRequest(new Read<String>() {

			@Override
			public String perform(ReadGraph graph) throws DatabaseException {
				BrowsingResource br = BrowsingResource.getInstance(graph);
				return graph.getPossibleRelatedValue(configuration, br.Button_Text);
			}
			
		});

		if(text != null) button.setText(text);

		String tooltip = Simantics.getSession().syncRequest(new Read<String>() {

			@Override
			public String perform(ReadGraph graph) throws DatabaseException {
				BrowsingResource br = BrowsingResource.getInstance(graph);
				return graph.getPossibleRelatedValue(configuration, br.Button_Tooltip);
			}
			
		});

		if(tooltip != null) button.setTooltipText(tooltip);
		
		ModelledIcon icon = Simantics.getSession().syncRequest(new Read<ModelledIcon>() {

			@Override
			public ModelledIcon perform(ReadGraph graph) throws DatabaseException {
				BrowsingResource br = BrowsingResource.getInstance(graph);
				Resource icon = graph.getPossibleObject(configuration, br.Button_Icon);
				if(icon == null) return null;
				else return graph.adapt(icon, ModelledIcon.class);
			}
			
		});
		
		if(icon != null) {
			ImageDescriptor iconDescriptor = icon.create();
			if (iconDescriptor != null) {
			    Image image = resourceManager.createImage(iconDescriptor);
			    button.setImage(image);
			} else {
			    ErrorLogger.defaultLogWarning("Could not load icon " + icon, null);
			}
		}
		
		ModelledAction modelledAction = Simantics.getSession().syncRequest(new Read<ModelledAction>() {

			@Override
			public ModelledAction perform(ReadGraph graph) throws DatabaseException {
				BrowsingResource br = BrowsingResource.getInstance(graph);
				Resource action = graph.getPossibleObject(configuration, br.Button_Action);
				if(action == null) return null;
				return graph.adapt(action, ModelledAction.class);
			}
			
		});

		final Runnable actionRunnable = modelledAction != null ? modelledAction.create(site, context, support) : null; 
		
		button.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetSelected(SelectionEvent e) {
				if(actionRunnable != null) actionRunnable.run();
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}
			
		});
		
		return button.getWidget();
		
	}
	
}
