/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.swt;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.viewers.IPostSelectionProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.simantics.Simantics;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.model.queries.IsNodeContextModifiable;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.management.ISessionContextProvider;
import org.simantics.ui.workbench.action.ChooseActionRequest;
import org.simantics.utils.datastructures.Function;
import org.simantics.utils.ui.ISelectionUtils;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * The default keyboard handler for {@link GraphExplorer}.
 * 
 * <p>
 * Carriage return presses are handled similarly to mouse double clicks. F2
 * initiates in-line editing of the selection when possible.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public class DefaultKeyListener extends KeyAdapter {

    private final ISessionContextProvider contextProvider;
    private final GraphExplorer          explorer;
    private final Function<String[]>      editingColumnResolver;

    /**
     * @param contextProvider
     * @param explorer
     * @param editingColumnResolver
     */
    public DefaultKeyListener(ISessionContextProvider contextProvider, GraphExplorer explorer, Function<String[]> editingColumnResolver) {
        assert contextProvider != null;
        assert explorer != null;
        assert editingColumnResolver != null;

        this.contextProvider = contextProvider;
        this.explorer = explorer;
        this.editingColumnResolver = editingColumnResolver;
    }

    @Override
    public void keyPressed(KeyEvent e) {
        if (e.keyCode == SWT.CR || e.keyCode == SWT.KEYPAD_CR) {
            if (explorer.isDisposed())
                return;

            ISessionContext sessionContext = contextProvider.getSessionContext();
            if (sessionContext == null)
                return;

            IPostSelectionProvider selectionProvider = (IPostSelectionProvider)explorer.getAdapter(IPostSelectionProvider.class);

            ISelection input = selectionProvider.getSelection();
            if (input.isEmpty())
                return;
//            Resource resource = ResourceAdaptionUtils.toSingleResource(selectionProvider.getSelection());
//            if (resource == null)
//                return;
            final String perspectiveId = WorkbenchUtils.getCurrentPerspectiveId();
            Control control = explorer.getControl();
            Shell shell = control.getShell();

            // Try the doubleClick-extensions
            sessionContext.getSession().asyncRequest(new ChooseActionRequest(shell, explorer, input, perspectiveId));
        } else if (e.keyCode == SWT.F2) {

            IPostSelectionProvider selectionProvider = (IPostSelectionProvider)explorer.getAdapter(IPostSelectionProvider.class);
            NodeContext context = ISelectionUtils.filterSingleSelection(selectionProvider.getSelection(), NodeContext.class);
            if (context == null || !isEditable(context))
                return;

            for (String column : editingColumnResolver.execute(context)) {
                // Prefix column key with '#' to indicate that
                // textual editing is preferred if supported.
                String error = explorer.startEditing(context, "#" + column); 
                if (error == null)
                    return;
            }
            return;
        }
    }

    private boolean isEditable(NodeContext context) {
        try {
            return Simantics.getSession().syncRequest(new IsNodeContextModifiable(context));
        } catch (DatabaseException e) {
            Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID,
                    "Failed to check node context modifiability, see exception for details.", e));
            return false;
        }
    }

}
