/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.ui;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.resource.ImageDescriptor;
import org.osgi.framework.Bundle;

public final class BundleUtils {

    /**
     * Returns an image descriptor for the image file at the given
     * plug-in relative path.
     *
     * @param pluginId the plugin to search
     * @param imageFilePath the path
     * @return the image descriptor
     */
    public static ImageDescriptor getImageDescriptorFromBundle(Bundle bundle, String imageFilePath) {
        if (bundle == null) {
            throw new IllegalArgumentException("null bundle");
        }
        if (imageFilePath == null) {
            throw new IllegalArgumentException("null image path");
        }

        // if the bundle is not ready then there is no image
        if (!isReady(bundle))
            return null;

        // look for the image (this will check both the plugin and fragment folders
        URL fullPathString = FileLocator.find(bundle, new Path(imageFilePath), null);
        if (fullPathString == null) {
            try {
                fullPathString = new URL(imageFilePath);
            } catch (MalformedURLException e) {
                return null;
            }
        }

        return ImageDescriptor.createFromURL(fullPathString);
    }

    /**
     * Returns an image descriptor for the image file at the given
     * plug-in relative path.
     *
     * @param pluginId the plugin to search
     * @param imageFilePath the path
     * @return the image descriptor
     */
    public static ImageDescriptor getImageDescriptorFromPlugin(String pluginId, String imageFilePath) {
        if (pluginId == null || imageFilePath == null) {
            throw new IllegalArgumentException();
        }
        Bundle bundle = Platform.getBundle(pluginId);
        return getImageDescriptorFromBundle(bundle, imageFilePath);
    }

    public static boolean isReady(Bundle bundle) {
        return bundle != null && isReady(bundle.getState());
    }
    
    public static boolean isReady(int bundleState) {
        return (bundleState & (Bundle.RESOLVED | Bundle.STARTING | Bundle.ACTIVE | Bundle.STOPPING)) != 0;
    }

    public static URL find(Bundle bundle, String path) {
        if (bundle == null)
            return null;
        return FileLocator.find(bundle, new Path(path), null);
    }

    public static URL find(String bundleId, String path) {
        return find(Platform.getBundle(bundleId), path);
    }

	public static File findFile(String bundleId, String path) throws IOException {
		URL url = find(bundleId, path);
		if (url == null)
		    return null;
		url = FileLocator.toFileURL(url);
		return new File(url.getPath());
	}

    /**
     * @since 1.31.0
     */
    public static File resolveWritableBundleFile(URL url) throws IOException {
        // This returns file, jar, http etc. - essentially resolves the bundle protocol
        URL resolved = FileLocator.resolve(url);
        if (resolved.getProtocol().equals("file")) {
            return new File(resolved.getPath());
        }
        return null;
    }

}
