/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures;

import org.simantics.utils.strings.AlphanumComparator;


/**
 * A generic Pair (2-tuple) structure for containing two object instances of
 * chosen types.
 * 
 * @param <T1> type of first element
 * @param <T2> type of second element
 */
public final class ComparablePair<T1 extends Comparable<T1>, T2 extends Comparable<T2>> implements Comparable<ComparablePair<T1, T2>> {

	public final T1 first;
    public final T2 second;
    private final int hash;

    public static <T1 extends Comparable<T1>, T2 extends Comparable<T2>> ComparablePair<T1, T2> make(T1 t1, T2 t2) {
        return new ComparablePair<T1, T2>(t1, t2);
    }

    public ComparablePair(T1 first, T2 second) {
    	if(first == null) throw new IllegalArgumentException("ComparablePair does not accept null values");
    	if(second == null) throw new IllegalArgumentException("ComparablePair does not accept null values");
        this.first = first;
        this.second = second;
        this.hash = makeHash();
    }
    
    @Override
    public boolean equals(Object obj) {
        if (obj == null)
            return false;
        if (!(obj.getClass().equals(this.getClass())))
            return false;
        ComparablePair<?, ?> other = (ComparablePair<?, ?>) obj;
        if (other.first != first && (other.first == null || !other.first.equals(first)))
            return false;
        if (other.second != second && (other.second == null || !other.second.equals(second)))
            return false;
        return true;
    }
    
    @Override
    public int hashCode() {
        return hash;
    }
    
    @Override
    public String toString() {
        return "<"+first+", "+second+">";
    }
    
    private int makeHash() {
        return (first == null ? 0 : first.hashCode()) + (second == null ? 0 : second.hashCode())*31;
    }

	@Override
	public int compareTo(ComparablePair<T1, T2> arg0) {
		int firstCompare = AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(first, arg0.first);
		if(firstCompare != 0) return firstCompare;
		else return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(second, arg0.second);
	}
	
}
