package org.simantics.ui.workbench.e4;

import java.util.Map;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;

import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.e4.ui.di.Focus;
import org.eclipse.e4.ui.di.Persist;
import org.eclipse.e4.ui.di.PersistState;
import org.eclipse.e4.ui.model.application.ui.basic.MPart;
import org.eclipse.e4.ui.workbench.modeling.EPartService;
import org.eclipse.e4.ui.workbench.modeling.ESelectionService;
import org.eclipse.swt.widgets.Composite;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.ui.workbench.IResourceEditorInput;

public abstract class E4ResourceEditorBase {

    @Inject
    private ESelectionService selectionService;

    @Inject
    private IEclipseContext eclipseContext;

    private MPart part;
    private boolean disposed;

    private Resource resource;
    private Variable variable;
    private String rvi;
    private Resource model;
    
    private IResourceEditorInput input;
    private E4ResourceEditorSupport support;

    @Inject
    public final void init(MPart mPart) throws NumberFormatException, DatabaseException {
        this.part = mPart;
        readTransientState();
        if (resource == null && model == null && rvi == null && variable == null)
            readPersistentState();

        deriveState();
        initImpl(part);

        if (!part.getTags().contains(EPartService.REMOVE_ON_HIDE_TAG)) {
            part.getTags().add(EPartService.REMOVE_ON_HIDE_TAG);
        }
    }

    public void initImpl(MPart part) { }

    private void deriveState() throws DatabaseException {
        boolean deriveVariable = variable == null && model != null;
        boolean deriveModelAndRVI = variable != null;
        boolean deriveModel = resource != null && model == null;

        if (deriveModel || deriveModelAndRVI || deriveVariable) {
            Simantics.getSession().syncRequest(new ReadRequest() {
                
                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    if (deriveVariable) {
                        if (rvi != null) {
                            Variable configuration = Variables.getConfigurationContext(graph, model);
                            RVI rrvi = RVI.fromResourceFormat(graph, rvi);
                            variable = rrvi.resolve(graph, configuration);
                        }
                        // Absolute URI
                        else {
                            variable = Variables.getVariable(graph, model);
                        }
                    }

                    if (deriveModelAndRVI) {
                        model = Variables.getPossibleModel(graph, variable);
                        rvi = variable.getRVI(graph).toString();
                    }

                    if (deriveModel) {
                        model = graph.syncRequest(new PossibleModel(resource));
                    }
                }
            });
        }

        Map<String, Object> transientData = part.getTransientData(); 
        transientData.put(E4ResourceEditorConstants.KEY_RESOURCE, resource);
        transientData.put(E4ResourceEditorConstants.KEY_MODEL, model);
        transientData.put(E4ResourceEditorConstants.KEY_VARIABLE, variable);
        transientData.put(E4ResourceEditorConstants.KEY_RVI, rvi);

        if (resource != null) {
            if (model != null) {
                input = new E4ResourceEditorInput2(resource, model, rvi);
            } else {
                input = new E4ResourceEditorInput(resource);
            }
        }
    }

    protected void activateValidation() {
        this.support = new E4ResourceEditorSupport(this, getInputValidator());
        this.support.activateValidation();
    }

    protected void deactivateValidation() {
        if (support != null) {
            support.dispose();
            support = null;
        }
    }

    /**
     * Override to define your own input resource editor input validator that
     * the view uses by default in {@link #initializeSupport()}.
     */
    protected ParametrizedRead<IResourceEditorInput, Boolean> getInputValidator() {
        return null;
    }

    private void readTransientState() throws DatabaseException {
        Map<String, Object> transientData = part.getTransientData();
        // Read input resource
        Object inputResource = transientData.get(E4ResourceEditorConstants.KEY_RESOURCE);
        if (inputResource != null && inputResource instanceof Resource)
            resource = (Resource) inputResource;

        // Read input RVI
        Object inputRVI = transientData.get(E4ResourceEditorConstants.KEY_RVI);
        if (inputRVI != null && inputRVI instanceof String)
            rvi = (String) inputRVI;

        Object inputModel = transientData.get(E4ResourceEditorConstants.KEY_MODEL);
        if (inputModel != null && inputModel instanceof Resource)
            model = (Resource) inputModel;

        // Read input variable
        Object inputVariable = transientData.get(E4ResourceEditorConstants.KEY_VARIABLE);
        if (inputVariable != null && inputVariable instanceof Variable)
            variable = (Variable) inputVariable;
    }

    private void readPersistentState() throws NumberFormatException, DatabaseException {
        Map<String, String> persistState = part.getPersistedState();
        SerialisationSupport support = Simantics.getSession().getService(SerialisationSupport.class);
        String inputResource = persistState.get(E4ResourceEditorConstants.KEY_RESOURCE);
        if (inputResource != null)
            resource = support.getResource(Long.parseLong(inputResource));

        // Read input RVI
        String inputRVI = persistState.get(E4ResourceEditorConstants.KEY_RVI);
        if (inputRVI != null)
            rvi = inputRVI;

        String inputModel = persistState.get(E4ResourceEditorConstants.KEY_MODEL);
        if (inputModel != null)
            model = support.getResource(Long.parseLong(inputModel));
    }

    @PostConstruct
    public void createEditor(Composite parent) {
        createPartControl(parent);
    }

    public abstract void createPartControl(Composite parent);

    @PersistState
    public void persistState() {
        System.out.println("persistState");
        Map<String, String> persistedState = part.getPersistedState();
        if (resource != null)
            persistedState.put(E4ResourceEditorConstants.KEY_RESOURCE, Long.toString(resource.getResourceId()));
        if (model != null)
            persistedState.put(E4ResourceEditorConstants.KEY_MODEL, Long.toString(model.getResourceId()));
        if (rvi != null)
            persistedState.put(E4ResourceEditorConstants.KEY_RVI, rvi);
    }

    @Persist
    public void persist() {
        System.out.println("persist");
    }

    @PreDestroy
    public void preDestroy() {
        System.out.println("dispose");
        disposed = true;
        dispose();
        deactivateValidation();
    }

    public abstract void dispose();

    public boolean isDisposed() {
        return disposed;
    }

    public abstract void setFocus();

    @Focus
    public void focus() {
        setFocus();
    }

    public MPart getPart() {
        return part;
    }

    public Resource getInputResource() {
        return resource;
    }

    public Variable getInputVariable() {
        return variable;
    }

    public String getInputRVI() {
        return rvi;
    }

    public Resource getInputModelResource() {
        return model;
    }

    public Resource assertInputModelResource() {
        if (model == null)
            throw new NullPointerException("Input model resource not defined, input resource is " + resource);
        return model;
    }

    public IResourceEditorInput getResourceInput() {
        return input;
    }

    public ESelectionService getSelectionService() {
        return selectionService;
    }

    public IEclipseContext getContext() {
        return eclipseContext;
    }

}
