/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench;

import org.eclipse.swt.widgets.Composite;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.event.ChangeEvent;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * This class inherits the Graph access setup behavior from
 * GraphAccessViewPart and in addition assumes that the view receives a
 * Resource ID as a parameter in the secondary ID of the view part.
 * 
 * <p>
 * To use this class all you need to do is call super.createPartControl in your
 * own createPartControl implementation. This will make sure a {@link Graph}
 * will be available directly after that for initializing the UI and its
 * contents.
 * </p>
 * 
 * <p>
 * To open a ResourceInputViewPart use:
 * {@link ResourceViewPartUtils#activateViewForResource(String, org.simantics.db.Session, Resource)}</code>
 * which should give the view the specified resource id as its secondary id.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public abstract class ResourceInputViewPart extends GraphAccessViewPart {

    private static final String NO_NAME        = "(no name)";

    private static final String MULTIPLE_NAMES = "(multiple names)";


    private ResourceInput       mainInput;

    private Resource            inputResource;

    private String              inputName      = NO_NAME;


    //----------------------------------------------------------------------
    // Getters

    public Resource getInputResource() {
        return inputResource;
    }

    public String getInputName() {
        return inputName;
    }


    //----------------------------------------------------------------------
    // Event handlers & initialization

    /**
     * Must be called after initializeGraph.
     */
    private void initializeInput() {
        // Get the main input from the view's secondary ID.
        String secondaryId = getViewSite().getSecondaryId();
        if (secondaryId == null)
            // No input given!
            return;

        mainInput = ResourceInput.unmarshall(secondaryId);
        try {
            inputResource = mainInput.toResource(getSession());
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
        }
    }


    /**
     * Initializes graph data access and view resource ID input structures.
     * 
     * <p>
     * This method is automatically called by
     * {@link #createPartControl(Composite)}. Override to perform own
     * graph-related initializations but be absolutely sure to call super the
     * first thing. Clients must not directly call this method.
     * </p>
     */
    @Override
    protected void initialize() {
        super.initialize();
        initializeInput();
    }

    /**
     * @return <code>true</code> if the input resource still exists, i.e.
     *         there are triples with the subject of the input resource.
     */
    protected boolean inputExists(ReadGraph g) throws DatabaseException {
        if (inputResource == null)
            return false;
        return g.hasStatement(inputResource);
    }

    @Override
    protected void cleanup() {
        inputResource = null;
        super.cleanup();
    }


    /**
     * @return the main input structure of this view which identifies a single
     *         input resource
     */
    public ResourceInput getMainInput() {
        return mainInput;
    }


    //----------------------------------------------------------------------
    // Override these if necessary:

    @Override
    protected String getTitleText() {
        return getInputName();
    }

    @Override
    protected String getTitleTooltip() {
        return getInputName();
    }


    /**
     * Extends {@link #update(GraphChangeEvent)} to handle disappearing inputs
     * by hiding the view.
     * 
     * @param event
     *            the received change event
     */
    @Override
    protected void update(ChangeEvent event) throws DatabaseException {
        ReadGraph g = event.getGraph();

        // If the input no longer exists, the editor should be closed since the
        // situation is analogous to having an IEditorPart input, such as a
        // file, deleted suddenly.
        if (!inputExists(g)) {
            WorkbenchUtils.hideView(getViewSite().getWorkbenchWindow(), this);
            return;
        }

        updateModelCache(g);
        reload(g);
    }

    private void updateModelCache(ReadGraph graph) {
        try {
        	Layer0 L0 = Layer0.getInstance(graph);
            inputName = graph.getPossibleRelatedValue(inputResource, L0.HasName);
            if (inputName == null)
                inputName = NO_NAME;
        } catch (DatabaseException e) {
            inputName = MULTIPLE_NAMES;
        }
    }

}
