/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench;

import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.Queries;
import org.simantics.db.exception.DatabaseException;


/**
 * A wrapper class for parsing and generating the information given to Simantics
 * views as a parameter through view secondary ID's.
 * 
 * <p>
 * To parse the input data (given in the secondary ID) just invoke:<br>
 * <code>new ResourceInput(secondaryIdString)</code>
 * </p>
 * 
 * <p>
 * To generate a secondary ID string based an input resource ID, invoke:<br>
 * <code>new ResourceInput(inputResourceId).marshall()</code>
 * </p>
 * 
 * @author Tuukka Lehtonen
 * 
 * @see ResourceViewPartUtils
 */
public class ResourceInput {

    private static final String SUFFIX_SEPARATOR = "@";

    private static final String EMPTY_SUFFIX = "";

    private final String        inputResourceId;

    private final String        suffix;

    /**
     * Construct a new input based on ID's.
     * Use this to serialize ID data into an input string.
     * 
     * @param randomAccessResourceId
     */
    public ResourceInput(String randomAccessResourceId) {
        this.inputResourceId = randomAccessResourceId;
        this.suffix = EMPTY_SUFFIX;
    }

    /**
     * Construct a new input based on ID's. Use this to serialize ID data into
     * an input string.
     * 
     * @param randomAccessResourceId
     * @param uniqueSuffix a unique suffix to add to the secondary ID or
     *        <code>null</code> to ignore the unique suffix
     */
    public ResourceInput(String randomAccessResourceId, String uniqueSuffix) {
        this.inputResourceId = randomAccessResourceId;
        this.suffix = (uniqueSuffix == null) ? EMPTY_SUFFIX : uniqueSuffix;
    }

    /**
     * Returns a new ResourceInput instance with the specified ID as
     * input. This method exists in order to make sure that the client can get a
     * clone of the original ResourceInput in every way except for the
     * input ID in a unified fashion.
     * 
     * @param newInputResourceId the new input resource ID
     * @return new ResourceInput instance with the specified input resource ID
     */
    public ResourceInput changeInput(String newRandomAccessResourceId) {
        if (suffix == EMPTY_SUFFIX)
            return new ResourceInput(newRandomAccessResourceId);
        return new ResourceInput(newRandomAccessResourceId, suffix);
    }

    public String getInputResourceId() {
        return inputResourceId;
    }

    public String getSuffix() {
        return suffix;
    }

    @Override
    public String toString() {
        return marshall();
    }

    /**
     * @param session the Session to use for initiating the
     *        transaction for retrieving the input Resource.
     * @return <code>null</code> if the random access reference of this
     *         ResourceInput is invalid.
     */
    public Resource toResource(Session session) throws DatabaseException {
        return session.syncRequest(Queries.resource(inputResourceId));
    }

    /**
     * @param input Input format:
     *        <code>client-id#input-resource-id[#unique-suffix]</code>
     * @throws IllegalArgumentException if input is invalid
     */
    public String marshall() {
        return String.format("%s@%s", inputResourceId, suffix);
    }

    /**
     * @param a previously marshalled ResourceInput using
     *        {@link #marshall()}
     * @throws IllegalArgumentException if input is invalid
     */
    public static ResourceInput unmarshall(String input) throws IllegalArgumentException {
        if (input == null)
            throw new IllegalArgumentException("null input");
        String[] parts = input.split(SUFFIX_SEPARATOR);
        if (parts.length < 1)
            throw new IllegalArgumentException("invalid input: " + input);

        // Get input resource random access id
        String id = parts[0];

        // Get arbitrary suffix
        String suffix = EMPTY_SUFFIX;
        if (parts.length > 1) {
            suffix = input.substring(id.length() + 1);
        }
        return new ResourceInput(id, suffix);
    }

    public static Resource unmarshallToResource(String input, Session session) throws DatabaseException {
        ResourceInput in = unmarshall(input);
        return in.toResource(session);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((inputResourceId == null) ? 0 : inputResourceId.hashCode());
        result = prime * result + ((suffix == null) ? 0 : suffix.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        final ResourceInput other = (ResourceInput) obj;
        if (inputResourceId == null) {
            if (other.inputResourceId != null)
                return false;
        } else if (!inputResourceId.equals(other.inputResourceId))
            return false;
        if (suffix == null) {
            if (other.suffix != null)
                return false;
        } else if (!suffix.equals(other.suffix))
            return false;
        return true;
    }

}
