package org.simantics.tests.modelled.junit.v2;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.junit.runner.Description;
import org.junit.runner.Result;
import org.junit.runner.notification.RunListener;
import org.junit.runner.notification.RunNotifier;
import org.junit.runners.ParentRunner;
import org.junit.runners.model.InitializationError;
import org.simantics.Simantics;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.testing.common.AcornTests;
import org.simantics.tests.modelled.utils.ModelledSTSSuite;
import org.simantics.tests.modelled.utils.STSSuiteTestCollector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ModelledSTSRunner extends ParentRunner<ModelledSTSSuiteRunner> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ModelledSTSRunner.class);
    
    public static final String INCLUSION_FILTER = "org.simantics.tests.modelled.singleTestIncludeFilter";
    public static final String EXCLUSION_FILTER = "org.simantics.tests.modelled.excludeFilter";
    
    private final List<ModelledSTSSuiteRunner> children;

    public ModelledSTSRunner(Class<?> testClass) throws InitializationError {
        super(testClass);

        try {
            initialize0();
            Collection<ModelledSTSSuite> suites = STSSuiteTestCollector.collectTests();
            
            // Filter exclusions
            String exclusionFilter = System.getProperty(EXCLUSION_FILTER);
            Collection<ModelledSTSSuite> filtered;
            if (exclusionFilter != null) {
                String[] filters = exclusionFilter.split(",");
                filtered = suites.stream().filter(s -> !startsWithAny(s, filters)).collect(Collectors.toList());
            } else {
                filtered = suites;
            }
            // Filter inclusions
//            String inclusionFilter = System.getProperty(INCLUSION_FILTER);
//            Collection<ModelledSTSSuite> included;
//            if (inclusionFilter != null) {
//                String[] filters = inclusionFilter.split(",");
//                included = filtered.stream().filter(s -> startsWithAny(s, filters)).collect(Collectors.toList());
//            } else {
//                included = filtered;
//            }
            
            // Sort by priority
//            List<ModelledSTSSuite> sorted = included.stream().sorted((o1, o2) -> Integer.compare(o1.getPriority(), o2.getPriority())).collect(Collectors.toList());
            List<ModelledSTSSuite> sorted = filtered.stream().sorted((o1, o2) -> Integer.compare(o1.getPriority(), o2.getPriority())).collect(Collectors.toList());
            
            
            children = new ArrayList<>(sorted.size());
            for (ModelledSTSSuite suite : sorted) {
                children.add(new ModelledSTSSuiteRunner(suite));
            }
        } catch (Exception e) {
            throw new InitializationError(e);
        }
    }

    private static boolean startsWithAny(ModelledSTSSuite suite, String[] filters) {
        for (String filter : filters) {
            if (suite.getName().contains(filter)) {
                return true;
            }
        }
        return false;
    }

    @Override
    protected List<ModelledSTSSuiteRunner> getChildren() {
        return children;
    }

    @Override
    protected Description describeChild(ModelledSTSSuiteRunner child) {
        return child.getDescription();
    }

    @Override
    public void run(RunNotifier notifier) {
        notifier.addListener(new RunListener() {

            @Override
            public void testRunFinished(Result result) throws Exception {
                deinitialize0();
            }
        });
        super.run(notifier);
    }

    @Override
    protected void runChild(ModelledSTSSuiteRunner child, RunNotifier notifier) {
        try {
            child.run(notifier);
        } finally {
            // Clear query cache
            Layer0Utils.queryDebugSupport("QueryControl.flush");
        }
        // TODO: Add coverage reporting to ModelledSTSRunner
//        CombinedCoverage cover = child.getCoverage();
//        CoverageBuilder b = new CoverageBuilder();
    }

    public void initialize() throws InitializationError {
    }

    public void deinitialize() throws Exception {
    }

    private void initialize0() throws Exception {
        AcornTests.newSimanticsWorkspace(null, null);
        org.simantics.debug.browser.internal.Activator.getDefault().startDebugServer();
        initialize();
    }

    private void deinitialize0() throws Exception {
        deinitialize();
        org.simantics.debug.browser.internal.Activator.getDefault().stopDebugServer();
        Simantics.shutdown(new NullProgressMonitor());
    }
}
