package org.simantics.tests.modelled;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.regex.Pattern;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.scl.compiler.commands.CommandSession;
import org.simantics.scl.compiler.commands.TestScriptExecutor;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.coverage.CombinedCoverage;
import org.simantics.scl.compiler.module.coverage.CoverageUtils;
import org.simantics.scl.compiler.module.options.ModuleCompilationOptions;
import org.simantics.scl.compiler.module.options.ModuleCompilationOptionsAdvisor;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.runtime.RuntimeModule;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.runtime.reporting.SCLReportingHandler;
import org.simantics.tests.modelled.ontology.TestsResource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestsGraphUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(TestsGraphUtils.class);

    private static final String STS_TEST_PREFIX = "STSTest";
    private static final String STS_SUITE_PREFIX = "STSSuite";
    private static final String STS_VARIABLE_PREFIX = "STSVariable";

    private TestsGraphUtils() {}

    public static Resource createSTSVariable(WriteGraph graph, Resource parent) throws DatabaseException {
        String name = NameUtils.findFreshEscapedName(graph, STS_VARIABLE_PREFIX, parent);
        Resource stsVariable = graph.newResource();

        Layer0 L0 = Layer0.getInstance(graph);
        TestsResource TESTS = TestsResource.getInstance(graph);

        graph.claim(parent, L0.ConsistsOf, L0.PartOf, stsVariable);
        graph.claim(stsVariable, L0.InstanceOf, TESTS.STSVariable);
        graph.claimLiteral(stsVariable, L0.HasName, name, Bindings.STRING);
        graph.claimLiteral(stsVariable, TESTS.STSVariable_definition, "", Bindings.STRING);
        return stsVariable;
    }

    public static Resource createSTSTest(WriteGraph graph, Resource parent) throws DatabaseException {
        String name = NameUtils.findFreshEscapedName(graph, STS_TEST_PREFIX, parent);
        Resource stsTest = graph.newResource();

        Layer0 L0 = Layer0.getInstance(graph);
        TestsResource TESTS = TestsResource.getInstance(graph);

        graph.claim(parent, L0.ConsistsOf, L0.PartOf, stsTest);
        graph.claim(stsTest, L0.InstanceOf, TESTS.STSTest);
        graph.claimLiteral(stsTest, L0.HasName, name, Bindings.STRING);
        graph.claimLiteral(stsTest, TESTS.STSTest_definition, "", Bindings.STRING);
        return stsTest;
    }

    public static Resource createSTSSuite(WriteGraph graph, Resource parent) throws DatabaseException {
        String name = NameUtils.findFreshEscapedName(graph, STS_SUITE_PREFIX, parent);
        Resource stsSuite = graph.newResource();

        Layer0 L0 = Layer0.getInstance(graph);
        TestsResource TESTS = TestsResource.getInstance(graph);

        graph.claim(parent, L0.ConsistsOf, L0.PartOf, stsSuite);
        graph.claim(stsSuite, L0.InstanceOf, TESTS.STSSuite);
        graph.claimLiteral(stsSuite, L0.HasName, name, Bindings.STRING);
        return stsSuite;
    }

    public static CombinedCoverage runSTSTestDefinition(String definition, List<Pattern> moduleNameFilters, SCLReportingHandler handler) throws IOException {

//        ModuleRepository repo = SCLOsgi.MODULE_REPOSITORY;
        ModuleRepository repo = new ModuleRepository(SCLOsgi.SOURCE_REPOSITORY);
        repo.setAdvisor(new ModuleCompilationOptionsAdvisor() {

            @Override
            public ModuleCompilationOptions getOptions(String moduleName) {
                boolean coverage = false;
                for (Pattern p : moduleNameFilters) {
                    if (p.matcher(moduleName.toLowerCase()).find()) {
                        coverage = true;
                        break;
                    }
                }
                return new ModuleCompilationOptions(coverage);
            }
        });

        CommandSession session = new CommandSession(repo, handler);
        TestScriptExecutor executor = new TestScriptExecutor(session, new BufferedReader(new StringReader(definition)), handler);
        try {
            executor.execute();
        } catch (Throwable t) {
            LOGGER.error("Failed to execute test definition:\n{}", definition, t);
        }
        Collection<RuntimeModule> runtimeModules = session.getRuntimeEnvironment().getRuntimeModules();
        List<Module> modules = new ArrayList<>(runtimeModules.size());
        for (RuntimeModule module : runtimeModules) {
            modules.add(module.getModule());
        }
        CombinedCoverage coverage = CoverageUtils.getCoverage(modules);
        return coverage;
    }

    public static byte[] stsTestContentDump(ReadGraph graph, Resource test) throws DatabaseException {
        TestsResource TESTS = TestsResource.getInstance(graph);
        String def = graph.getRelatedValue(test, TESTS.STSTest_definition, Bindings.STRING);
        return def.getBytes(StandardCharsets.UTF_8);
    }

}
