package org.simantics.structural2.scl;

import java.util.Collections;
import java.util.Map;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.ResourceRead2;
import org.simantics.db.common.request.RuntimeEnvironmentRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.scl.AbstractExpressionCompilationContext;
import org.simantics.db.layer0.scl.AbstractExpressionCompilationRequest;
import org.simantics.db.layer0.util.RuntimeEnvironmentRequest2;
import org.simantics.layer0.Layer0;
import org.simantics.scl.compiler.elaboration.expressions.EApply;
import org.simantics.scl.compiler.elaboration.expressions.EConstant;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.elaboration.modules.SCLValue;
import org.simantics.scl.compiler.environment.Environment;
import org.simantics.scl.compiler.environment.Environments;
import org.simantics.scl.compiler.runtime.RuntimeEnvironment;
import org.simantics.scl.compiler.top.SCLExpressionCompilationException;
import org.simantics.scl.compiler.types.Type;
import org.simantics.structural2.scl.AbstractCompileStructuralValueRequest.CompilationContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Compiles an SCL expression that is attached to a literal
 * whose parent is a component that is a part of a component type.
 * 
 * @author Antti Villberg
 */
public abstract class AbstractCompileStructuralValueRequest extends AbstractExpressionCompilationRequest<CompilationContext, Object> {
    
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractCompileStructuralValueRequest.class);

    protected final Resource relation;
    
    public static class CompilationContext extends AbstractExpressionCompilationContext {
        public final Map<String, ComponentTypeProperty> propertyMap;
        
        public CompilationContext(RuntimeEnvironment runtimeEnvironment,
                Map<String, ComponentTypeProperty> propertyMap) {
            super(runtimeEnvironment);
            this.propertyMap = propertyMap;
        }
    }
    
    public AbstractCompileStructuralValueRequest(Resource relation) {
        this.relation = relation;
    }

    @Override
    abstract protected String getExpressionText(ReadGraph graph) throws DatabaseException;
    
    /*
     * @throws DatabaseException is an index root could not be found
     */
    abstract protected Resource getIndexRoot(ReadGraph graph) throws DatabaseException;
    /*
     * Does not throw if the graph is valid.
     * 
     * @return null if component type was not availble
     * 
     */
    abstract protected Resource getComponentType(ReadGraph graph) throws DatabaseException;

    protected Resource getPossibleIndexRoot(ReadGraph graph) throws DatabaseException {
        try {
            return getIndexRoot(graph);
        } catch (DatabaseException e) {
            LOGGER.error("Error while resolving index root during structural value compilation", e);
            return null;
        }
    }
    
    @Override
    protected Type getExpectedType(ReadGraph graph, CompilationContext context)
            throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        String valueType = graph.getPossibleRelatedValue(relation, L0.RequiresValueType, Bindings.STRING);
        if(valueType != null) {
            Resource relationIndexRoot = graph.syncRequest(new PossibleIndexRoot(relation));
            RuntimeEnvironment relationRuntimeEnvironment = relationIndexRoot != null ? graph.syncRequest(new RuntimeEnvironmentRequest(relationIndexRoot)) : context.runtimeEnvironment;
            try {
                return Environments.getType(relationRuntimeEnvironment.getEnvironment(), valueType);
            } catch (SCLExpressionCompilationException e) {
                e.printStackTrace();
            }
        }
        return super.getExpectedType(graph, context);
    }
    
    

    @Override
    protected CompilationContext getCompilationContext(ReadGraph graph)
            throws DatabaseException {
        Resource indexRoot = getIndexRoot(graph);
        Resource componentType = getComponentType(graph);
        if(componentType == null) {
            RuntimeEnvironment runtimeEnvironment = graph.syncRequest(new RuntimeEnvironmentRequest(indexRoot));
            return new CompilationContext(runtimeEnvironment, Collections.<String,ComponentTypeProperty>emptyMap());
        } else {
	        return graph.syncRequest(new ResourceRead2<CompilationContext>(componentType, indexRoot) {
	            @Override
	            public CompilationContext perform(ReadGraph graph)
	                    throws DatabaseException {
	                RuntimeEnvironment runtimeEnvironment = graph.syncRequest(new RuntimeEnvironmentRequest2(resource, resource2));
	                Map<String, ComponentTypeProperty> propertyMap =
	                        graph.syncRequest(new ReadComponentTypeInterfaceRequest(resource, runtimeEnvironment.getEnvironment()),
	                                TransientCacheListener.<Map<String, ComponentTypeProperty>>instance());
	                return new CompilationContext(runtimeEnvironment, propertyMap);
	            }
	        });
        }
    }

    @Override
    protected Type getContextVariableType() {
        return VARIABLE; 
    }

    protected static Expression accessInputVariable(Environment environment,
            org.simantics.scl.compiler.elaboration.expressions.Variable contextVariable) {
        SCLValue variableParentFunction = environment.getValue(VARIABLE_PARENT);
        return new EApply(new EConstant(variableParentFunction),
                new EApply(new EConstant(variableParentFunction),
                        new EVariable(contextVariable)));
    }
    
    @Override
    protected Expression getVariableAccessExpression(
            ReadGraph graph,
            CompilationContext context,
            org.simantics.scl.compiler.elaboration.expressions.Variable contextVariable,
            String name) throws DatabaseException {
        ComponentTypeProperty property = context.propertyMap.get(name);
        if(property != null) {
            Environment environment = context.runtimeEnvironment.getEnvironment();
            return getPropertyFlexible(environment,
                    accessInputVariable(environment, contextVariable),
                    name,
                    property.type);
        }
        else if(name.equals("input")) {
            Environment environment = context.runtimeEnvironment.getEnvironment();
            return accessInputVariable(environment, contextVariable);
        }
        else if(name.equals("self"))
            return new EVariable(contextVariable);
        else
            return null;
    }
    
    @Override
    protected String getContextDescription(ReadGraph graph) throws DatabaseException {
        Resource componentType = getComponentType(graph);
        if(componentType != null) {
            String uri = graph.getPossibleURI(componentType);
            if(uri != null)
                return uri;
        }
        // OK, no component type - report index root then
        Resource indexRoot = getPossibleIndexRoot(graph);
        if(indexRoot != null) {
            String uri = graph.getPossibleURI(indexRoot);
            if(uri != null)
                return uri;
        }
        // OK, nothing - report default
        return super.getContextDescription(graph);
    }
    
}
