/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.simulator.variable;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.type.Datatype;
import org.simantics.simulator.variable.exceptions.InvalidPathException;
import org.simantics.simulator.variable.exceptions.NoSuchNodeException;
import org.simantics.simulator.variable.exceptions.NoValueException;
import org.simantics.simulator.variable.exceptions.NodeManagerException;
import org.simantics.simulator.variable.exceptions.NotInRealmException;

/**
 * {@code NodeManager} provides access to a local simulator
 * or a similar entity. Because simulation may be ongoing, access
 * is restricted to a certain realm.
 * 
 * @author Hannu Niemist&ouml;
 * @author Antti Villberg
 */
public interface NodeManager<Node> {

    public static final Variant PENDING_NODE_VALUE = new Variant();

    // --- Do not require a realm access ---

    /**
     * The realm of the node manager. Almost all other methods
     * of this class must be called inside this realm.
     */
    Realm getRealm();

    /**
     * Returns the name of the node. This method does not require 
     * that caller is in realm.
     */
    String getName(Node node);

    /**
     * Adds a listener to a certain node. The new listener is called as
     * soon as possible (for example before simulator takes the next simulation
     * step). After the first call, it is called always the node value 
     * or structure may have changed. This can be called outside of the realm.
     */
    void addNodeListener(Node node, Runnable listener);

    /**
     * Removes previously added listener. This can be called outside of
     * the realm.
     */
    void removeNodeListener(Node node, Runnable listener);

    // --- Require a realm access ---

    /**
     * @return {@code null} if node cannot be found, otherwise a node with the given path
     * @throws InvalidPathException if the path is not in a valid path format
     * @throws NotInRealmException if not synchronized to the realm
     */
    Node getNode(String path) throws NodeManagerException;	
    /**
     * @return {@code null} if node cannot be found, otherwise a child node with the given name
     * @throws NotInRealmException if not synchronized to the realm
     */
    Node getChild(Node node, String name) throws NodeManagerException;
    Node getProperty(Node node, String name) throws NodeManagerException;
    List<String> getChildNames(Node node) throws NodeManagerException;
    List<String> getPropertyNames(Node node) throws NodeManagerException;
    List<Node> getChildren(Node node) throws NodeManagerException;
    List<Node> getProperties(Node node) throws NodeManagerException;

    /**
     * @throws NoValueException if the node has no value (and therefore no datatype)
     * @throws NotInRealmException if not synchronized to the realm
     */
    Datatype getDatatype(Node node) throws NodeManagerException;
    /**
     * @throws NoValueException if the node has no value
     * @throws BindingException if the value can not be bound to the given binding
     * @throws NotInRealmException if not synchronized to the realm
     */
    Object getValue(Node node, Binding binding) throws NodeManagerException, BindingException;
    /**
     * A variant of {@link #getValue(Object, Binding)} that uses
     * a binding chosen by the node manager.
     */
    Variant getValue(Node node) throws NodeManagerException;
    /**
     * @throws NoSuchNodeException if the property does not exist
     * @throws NoValueException if the property has no value
     * @throws NotInRealmException if not synchronized to the realm
     * @throws BindingException if the value can not be bound to the given binding
     */
    Object getValue(Node node, String property, Binding binding) throws NodeManagerException, BindingException;
    /**
     * A variant of {@link #getValue(Object, String, Binding)} that uses
     * a binding chosen by the node manager.
     */
    Variant getValue(Node node, String property) throws NodeManagerException;	

    /**
     * @throws BindingException if the value can not be bound to the given binding
     * @throws NoValueException if the property has no value that could be assigned
     * @throws NotInRealmException if not synchronized to the realm
     */
    void setValue(Node node, Object value, Binding binding) throws NodeManagerException, BindingException;
    /**
     * @throws BindingException if the value can not be bound to the given binding
     * @throws NoSuchNodeException if the property does not exist
     * @throws NoValueException if the property has no value that could be assigned
     * @throws NotInRealmException if not synchronized to the realm
     */
    void setValue(Node node, String property, Object value, Binding binding) throws NodeManagerException, BindingException;

    /**
     * Asks the full URI of a property node. The parent of the property is also given as a parameter.
     * This is an optional method, NodeManager does not have to implement it for all nodes.
     */
    String getPropertyURI(Node parent, Node property);

    /**
     * Asks for the classifications of a property node.
     * This is an optional method, NodeManager does not have to implement it for all nodes.
     * A default implementation should just return {@link Collections#emptySet()}.
     * Classifications can be any strings, however a recommended interpretation is to return
     * the URIs of the primary ontological types that this node corresponds to.
     * 
     * @param node the node to classify
     * @return classifications of the node, empty set if the node has no classifications
     */
    Set<String> getClassifications(Node node) throws NodeManagerException;

}