package org.simantics.scl.compiler.elaboration.expressions.list;

import java.util.Set;

import org.simantics.scl.compiler.elaboration.contexts.SimplificationContext;
import org.simantics.scl.compiler.elaboration.contexts.TranslationContext;
import org.simantics.scl.compiler.elaboration.contexts.TypingContext;
import org.simantics.scl.compiler.elaboration.expressions.Case;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.elaboration.expressions.Variable;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.types.TMetaVar;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.compiler.types.kinds.Kinds;

public class ListGenerator extends ListQualifier {
    public Expression pattern;
    public Expression value;
    
    public ListGenerator(Expression pattern, Expression value) {
        this.pattern = pattern;
        this.value = value;
    }
    
    @Override
    public void checkType(TypingContext context) {
        TMetaVar componentType = Types.metaVar(Kinds.STAR);
        value = value.checkType(context, Types.apply(Types.LIST, componentType));
        pattern = pattern.checkTypeAsPattern(context, componentType);
    }

    @Override
    public CompiledQualifier compile(SimplificationContext context) {
        if(pattern instanceof EVariable)
            return new CompiledQualifier(value, pattern);
        else {
            Set<Variable> variables = pattern.getFreeVariables();
            Variable[] variableArray = variables.toArray(new Variable[variables.size()]);
            Expression[] variableExps = new Expression[variableArray.length];
            for(int i=0;i<variableArray.length;++i)
                variableExps[i] = new EVariable(variableArray[i]);
            Expression newPattern = context.tuple(variableExps);
            
            EVariable blank = context.blank();
            blank.getVariable().setType(pattern.getType());
            
            return new CompiledQualifier(
                    context.concatMap(
                            context.lambda(
                                    new Case(pattern, context.singletonList(newPattern.copy())),
                                    new Case(blank, context.emptyList(newPattern.getType()))
                                    ), 
                            value
                            ), 
                    newPattern);
        }
    }

    @Override
    public void resolve(TranslationContext context) {
        value = value.resolve(context);
        pattern = pattern.resolveAsPattern(context);
    }

    @Override
    public void setLocationDeep(long loc) {
        if(location == Locations.NO_LOCATION) {
            location = loc;
            pattern.setLocationDeep(loc);
            value.setLocationDeep(loc);
        }
    }
    
    @Override
    public void accept(ListQualifierVisitor visitor) {
        visitor.visit(this);
    }
    
    @Override
    public ListQualifier accept(ListQualifierTransformer transformer) {
        return transformer.transform(this);
    }
}
