/*******************************************************************************
 * Copyright (c) 2007 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project.management;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.TreeMap;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.project.ontology.ProjectResource;

/**
 * Project specification.
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class ProjectSpec {
 
	/** Display Name */
	String name;
	
	/** Project database unique identifier */
	String URI;
	
	/** Feature specification of the project. The list is sorted by versionedId */
	List<FeatureSpec> features;
	transient List<FeatureSpec> features_read_only_view;
	transient int hash;
	
	/**
	 * Create a new project specification
	 * 
	 * @param name optional Name
	 * @param URI project unique identifier
	 * @param features features of the project
	 */
	public ProjectSpec(String name, String URI, List<FeatureSpec> features) {
		this.name = name;
		this.URI = URI;
		TreeMap<String, FeatureSpec> sortedList = new TreeMap<String, FeatureSpec>();
		for (FeatureSpec fs : features) sortedList.put(fs.versionedId, fs);
		this.features = new ArrayList<FeatureSpec>(sortedList.values());
		this.features_read_only_view = Collections.unmodifiableList(this.features);
		hash = URI.hashCode();
		for (FeatureSpec fs : this.features) {
			hash = 13*hash + fs.hashCode();
		}
	}	

	public String getName() {
		return name;
	}

	public String getURI() {
		return URI;
	}
	
	@Override
	public int hashCode() {
		return hash;
	}
	
	@Override
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (obj instanceof ProjectSpec == false) return false;
		if (obj==this) return true;
		ProjectSpec other = (ProjectSpec) obj;
		if (!other.URI.equals(URI)) return false;
		if (other.features.size() != features.size()) return false;
		int c = features.size();
		for (int i=0; i<c; i++) {
			if (!features.get(i).equals(other.features.get(i))) return false;
		}
		return true;
	}

	/**
	 * Get a view to projects features
	 * 
	 * @return features, list sorted by versionedId
	 */
	public List<FeatureSpec> getFeatures() {
		return features;
	}
	
	/** A query that reads all bundles from the database, result ordered by URI */
	static Read<List<ProjectSpec>> QUERY = new Read<List<ProjectSpec>>() {
		@Override
		public List<ProjectSpec> perform(ReadGraph g) throws DatabaseException {
			ProjectResource pr = ProjectResource.getInstance(g);
			Layer0 L0 = Layer0.getInstance(g);
	        Resource root = g.getResource("http://Projects");
	        TreeMap<String, ProjectSpec> result = new TreeMap<String, ProjectSpec>();
	        TreeMap<String, FeatureSpec> features = new TreeMap<String, FeatureSpec>();
			for (Resource r : g.getObjects(root, L0.ConsistsOf) ) {
				String URI = g.getURI( r );
				String name = g.getPossibleRelatedValue(r, L0.HasName);
				
				features.clear();				
				for (Resource r2 : g.getObjects(r, pr.HasFeature)) {
					FeatureSpec spec = new FeatureSpec();
					spec.versionedId = g.getPossibleRelatedValue(r2, pr.HasGroupId, Bindings.STRING);
					spec.name = g.getPossibleRelatedValue(r2, L0.HasName, Bindings.STRING);
					Boolean b = g.getPossibleRelatedValue(r2, pr.IsRequired, Bindings.BOOLEAN);
					spec.required = b!=null ? b : true;					
					features.put(spec.versionedId, spec);
				}
				
				ProjectSpec spec = new ProjectSpec(name, URI, new ArrayList<FeatureSpec>(features.values()));
				result.put(URI, spec);
			}
			return new ArrayList<ProjectSpec>(result.values());
		}
	};
		
	
}

