/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.project;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.project.exception.ProjectException;
import org.simantics.project.features.registry.GroupReference;
import org.simantics.project.features.registry.IProjectFeatureExtension;
import org.simantics.project.features.registry.IProjectFeatureRegistry;
import org.simantics.project.features.registry.InjectedDependency;
import org.simantics.project.features.registry.ProjectFeatureReference;
import org.simantics.project.internal.Activator;
import org.simantics.project.ontology.ProjectResource;
import org.simantics.utils.datastructures.MapSet;

/**
 * A facade for handling project feature -related tasks.
 * 
 * @author Tuukka Lehtonen
 */
public class ProjectFeatures {

    /**
     * @return the singleton project feature registry service
     */
    public static IProjectFeatureRegistry getRegistry() {
        return Activator.getDefault().getProjectFeatureRegistry();
    }

    /**
     * @return
     */
    public static IProjectFeatureExtension[] getAllProjectFeatures() {
        return getRegistry().getExtensions();
    }

    /**
     * @return
     */
    public static Collection<IProjectFeatureExtension> getPublishedProjectFeatures() {
        IProjectFeatureExtension[] extensions = getAllProjectFeatures();
        Collection<IProjectFeatureExtension> result = new ArrayList<IProjectFeatureExtension>();
        for (IProjectFeatureExtension ext : extensions) {
            if (ext.isPublished())
                result.add(ext);
        }
        return result;
    }

    /**
     * @return
     */
    public static Set<GroupReference> getInstallGroupsOfPublishedFeatures() {
        return getInstallGroups( getPublishedProjectFeatures() );
    }

    /**
     * @param extensions
     * @return
     */
    public static Set<GroupReference> getInstallGroups(Collection<IProjectFeatureExtension> extensions) {
        Set<GroupReference> result = new TreeSet<GroupReference>();
        for (IProjectFeatureExtension ext : extensions)
            for (GroupReference grp : ext.installGroups())
                result.add(grp);
        return result;
    }

    /**
     * @param filter
     * @return
     */
    public static Collection<IProjectFeatureExtension> filterPublishedFeatures(GroupFilter filter) {
        return filterExtensions( getPublishedProjectFeatures(), filter );
    }

    private static Set<IProjectFeatureExtension> filterExtensions(Collection<IProjectFeatureExtension> extensions,
            GroupFilter filter) {
        Set<IProjectFeatureExtension> included = new HashSet<IProjectFeatureExtension>();
        for (IProjectFeatureExtension ext : extensions)
            if (acceptExtension(ext, filter))
                included.add(ext);
        return included;
    }

    private static boolean acceptExtension(IProjectFeatureExtension extension, GroupFilter filter) {
        for (GroupReference grp : extension.installGroups())
            if (filter.accept(grp))
                return true;
        return false;
    }

    /**
     * @param g
     * @param project
     * @return
     * @throws DatabaseException
     * @throws ProjectException 
     */
    public static Collection<IProjectFeatureExtension> getTopologicallySortedFeatures(GroupFilter filter) throws DatabaseException, ProjectException {
        return new FeatureWalker().getSortedFeatures(filter);
    }

    /**
     * @param g
     * @param project
     * @return
     * @throws DatabaseException
     * @throws ProjectException 
     */
    public static Collection<IProjectFeatureExtension> getTopologicallySortedFeatures(ReadGraph g, Resource project) throws DatabaseException, ProjectException {
        return new FeatureWalker().getSortedFeatures(g, project);
    }

    private static class FeatureWalker {

        private Map<String, IProjectFeatureExtension>                            byId;
        private final MapSet<IProjectFeatureExtension, IProjectFeatureExtension> required = new MapSet.Hash<IProjectFeatureExtension, IProjectFeatureExtension>();

        private IProjectFeatureExtension getExtension(ProjectFeatureReference ref) {
            return byId.get(ref.id);
        }

        public Collection<IProjectFeatureExtension> getSortedFeatures(ReadGraph g, Resource project) throws DatabaseException, ProjectException {
            Set<GroupReference> projectDefinedFeatures = loadFeatureReferences(g, project);
            return getSortedFeatures( GroupFilters.includesVersion( projectDefinedFeatures ) );
        }

        public Collection<IProjectFeatureExtension> getSortedFeatures(GroupFilter filter) throws DatabaseException, ProjectException {
            // Create ID -> Extension cache
            IProjectFeatureExtension[] allExtensions = getAllProjectFeatures();
            byId = createIdMap(allExtensions);

            // Find which extensions are to be loaded for the feature contexts.
            Set<IProjectFeatureExtension> included = filterExtensions(Arrays.asList(allExtensions), filter);

            // Add all required extensions
            Collection<IProjectFeatureExtension> required = requiredExtensions(allExtensions, included);

            // Sort topologically
            return sortTopologically(required);
        }

        private boolean deepRequires(Set<IProjectFeatureExtension> visited, IProjectFeatureExtension e1, IProjectFeatureExtension e2) {
            // Prevent eternal looping in cyclic cases
            // which are obviously erroneous definitions
            // but we don't want to get stuck anyway.
            if (visited.add(e1)) {
                // Check direct requires first.
                Set<IProjectFeatureExtension> req = required.getValues(e1);
                if (req == null)
                    return false;
                if (req.contains(e2))
                    return true;
                // Check transitively requirements
                for (IProjectFeatureExtension r : req) {
                    if (deepRequires(visited, r, e2))
                        return true;
                }
            }
            return false;
        }

        private void requiresDFS(IProjectFeatureExtension ext, ArrayList<IProjectFeatureExtension> result, Set<IProjectFeatureExtension> visited) {
            if(visited.add(ext)) {
                Set<IProjectFeatureExtension> reqs = required.getValues(ext);
                if(reqs != null) {
                    for(IProjectFeatureExtension req : reqs) {
                        requiresDFS(req, result, visited);
                    }
                }
                result.add(ext);
            }
        }

        private Collection<IProjectFeatureExtension> sortTopologically(Collection<IProjectFeatureExtension> toSort) {
            ArrayList<IProjectFeatureExtension> result = new ArrayList<>();
            Set<IProjectFeatureExtension> visited = new HashSet<>();
            for(IProjectFeatureExtension ext : toSort) {
                requiresDFS(ext, result, visited);
            }
            return result;
        }

        private Collection<IProjectFeatureExtension> requiredExtensions(IProjectFeatureExtension[] allExtensions, Collection<IProjectFeatureExtension> includedExtensions) throws ProjectException {
            Collection<IProjectFeatureExtension> result = new ArrayList<IProjectFeatureExtension>();
            Set<IProjectFeatureExtension> visited = new HashSet<IProjectFeatureExtension>();
            for (IProjectFeatureExtension ext : includedExtensions) {
                walkRequired(result, visited, ext);
            }

            Set<String> includedProjectFeatureIds = new HashSet<String>();
            for (IProjectFeatureExtension ext : result) {
                includedProjectFeatureIds.add(ext.getId());
            }

            // Find all injected dependencies!
            boolean changed = false;
            Set<IProjectFeatureExtension> injectionsProcessedFor = new HashSet<IProjectFeatureExtension>();
            do {
                changed = false;
                for (IProjectFeatureExtension ext : allExtensions) {
                    if (injectionsProcessedFor.contains(ext))
                        continue;

                    for (InjectedDependency injection : ext.injections()) {
                        if (includedProjectFeatureIds.contains(injection.to.id)) {
                            injectionsProcessedFor.add(ext);
                            IProjectFeatureExtension injectionTargetExt = byId.get(injection.to.id);
                            if (injectionTargetExt != null) {
                                changed = true;
                                includedProjectFeatureIds.add(ext.getId());
                                if(!result.contains(ext))
                                    result.add(ext);
                                required.add(injectionTargetExt, ext);
                            }
                        }
                    }
                }
            } while (changed);

            return result;
        }

        private void walkRequired(Collection<IProjectFeatureExtension> result, Set<IProjectFeatureExtension> visited, IProjectFeatureExtension extension) throws ProjectException {
            if (visited.add(extension)) {
                result.add(extension);
                for (ProjectFeatureReference ref : extension.requires()) {
                    IProjectFeatureExtension reqExt = getExtension(ref);
                    if (reqExt != null) {
                        required.add(extension, reqExt);
                        walkRequired(result, visited, reqExt);
                    } else if (!ref.optional) {
                        // This requirement was not optional, must report error!
                        throw new ProjectException("Missing org.simantics.project.feature extension with id '" + ref.id + "' required by feature '" + extension.getId() + "'");
                    }
                }
            }
        }
    }

    private static Set<GroupReference> loadFeatureReferences(ReadGraph graph, Resource project) throws DatabaseException {
        ProjectResource PROJ = ProjectResource.getInstance(graph);
        Set<GroupReference> features = new HashSet<GroupReference>();
        for (Resource fs : graph.getObjects(project, PROJ.HasFeature)) {
            String refStr = graph.getRelatedValue(fs, PROJ.HasGroupId, Bindings.STRING);
            try {
                GroupReference ref = parseFeatureReference(refStr);
                features.add(ref);
            } catch (IllegalArgumentException e) {
                // Parse error!
                e.printStackTrace();
            }
        }

        // See http://dev.simantics.org/index.php/Project_Development#Omnipresent_Project_Features
        features.add(GroupReference.OMNIPRESENT);

        return features;
    }

    private static Map<String, IProjectFeatureExtension> createIdMap(IProjectFeatureExtension[] extensions) {
        Map<String, IProjectFeatureExtension> byId = new HashMap<String, IProjectFeatureExtension>();
        for (IProjectFeatureExtension ext : extensions) {
            IProjectFeatureExtension e = byId.put(ext.getId(), ext);
            if (e != null) {
                Activator.getDefault().getLog().log(
                        new Status(IStatus.WARNING, Activator.PLUGIN_ID,
                                "Multiple org.simantics.project.feature extensions defined with id '" + ext.getId()
                                        + "'.\nprevious: " + e + "\nnew: " + ext)
                        );
            }
        }
        return byId;
    }

    static String ID_PATTERN_STRING = "[a-zA-Z_0-9]+(?:\\.[a-zA-Z_0-9]+)*";
    static String VERSION_PATTERN_STRING = ".*";
    static Pattern POSSIBLY_VERSIONED_ID_PATTERN = Pattern.compile("(" + ID_PATTERN_STRING + ")(?:/(" + VERSION_PATTERN_STRING + "))?");

    /**
     * @param reference
     * @return
     * @throws IllegalArgumentException if the reference string cannot be parsed
     */
    private static GroupReference parseFeatureReference(String reference) {
        Matcher m = POSSIBLY_VERSIONED_ID_PATTERN.matcher(reference);
        if (m.matches()) {
            return new GroupReference(m.group(1), m.group(2));
        } else {
            throw new IllegalArgumentException("could not parse feature reference " + reference);
        }
    }

}
