/*******************************************************************************
 * Copyright (c) 2012, 2013 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.graph.rules.range;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.simantics.objmap.exceptions.MappingException;


/**
 * Accessor for mapped value. Uses two methods:
 * - Getter: returns the current value.
 * - Setter: sets the current value. The value may be null. (if setter parameter is primitive, null value is not mapped).
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 * @param <T>
 */
public class GetSetValueAccessor<Range,T>  implements IRangeAccessor<Range,T> {
	
	private Method getter;
	private Method setter;
	private boolean primitive;
	
	public GetSetValueAccessor(Method getter, Method setter) {
		this.getter = getter;
		this.setter = setter;
		this.primitive = setter.getParameterTypes()[0].isPrimitive();
	}
	
	@SuppressWarnings("unchecked")
	public T get(Range element) throws MappingException {
		try {
			return (T) getter.invoke(element);
		} catch (IllegalArgumentException e) {
			throw new MappingException(e);
		} catch (IllegalAccessException e) {
			throw new MappingException(e);
		} catch (InvocationTargetException e) {
			throw new MappingException(e.getCause());
		}
	};

	@Override
	public boolean set(Range element, T value)
			throws MappingException {
		if (value == null && primitive)
			return false;
		if (equal(get(element),value))
			return false;
		try {
			setter.invoke(element, value);
		} catch (IllegalArgumentException e) {
			throw new MappingException(e);
		} catch (IllegalAccessException e) {
			throw new MappingException(e);
		} catch (InvocationTargetException e) {
			throw new MappingException(e.getCause());
		}
		return true;
		
	}
	
	private boolean equal(Object v1, Object v2) {
		if (v1 == null) {
			if (v2 == null)
				return true;
			return false;
		} else if (v2 == null) {
			return false;
		}
		return v1.equals(v2);
	}
}
