/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.graph.rules.domain;

import java.util.Arrays;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.OptionalBinding;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.BindingException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.DoesNotContainValueException;
import org.simantics.db.exception.ServiceException;
import org.simantics.layer0.Layer0;
import org.simantics.objmap.exceptions.MappingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Accesses a value attached to the element by given functional relation.
 * @author Hannu Niemist�
 */
public class RelatedValueAccessor implements IDomainAccessor<Resource,Object> {

    static Logger LOGGER = LoggerFactory.getLogger(RelatedValueAccessor.class);
    
	Resource relation;
	Resource valueType;
	Binding valueBinding;

	public RelatedValueAccessor(Resource relation, Resource valueType, Binding valueBinding) {
		this.relation = relation;
		this.valueType = valueType;
		this.valueBinding = valueBinding;
	}

	@Override
	public Object get(ReadGraph g, Resource element) throws MappingException {
		try {
		    LOGGER.trace("        RelatedValueAccessor.get");
			Resource valueResource = g.getPossibleObject(element, relation);
			if(valueResource == null)
				return null;
			return getValue(g, valueResource);
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
	}

	@Override
	public boolean set(WriteGraph g, Resource element, Object value)
			throws MappingException {
		try {
		    LOGGER.trace("        RelatedValueAccessor.set");
		    Statement valueStatement = g.getPossibleStatement(element, relation);
			if(valueStatement == null) {
				if(value == null)
					return false;
				Resource valueResource = g.newResource();
				g.claim(valueResource, Layer0.getInstance(g).InstanceOf, null,
						valueType);
				g.claim(element, relation, valueResource);
				claimValue(g, valueResource, value);
				
				return true;
			}
			else {
				if(value == null) {
					if (!valueStatement.isAsserted(element)) {
						g.deny(valueStatement.getObject());
						return true;
					} else {
						return false;
					}
				}				
				Object currentValue = getValue(g, valueStatement.getObject());
				if(equals(currentValue,value))
					return false;
				if (!valueStatement.isAsserted(element))
					g.claimValue(valueStatement.getObject(), value);
				else {
					Resource valueResource = g.newResource();
					g.claim(valueResource, Layer0.getInstance(g).InstanceOf, null,
							valueType);
					g.claim(element, relation, valueResource);
					claimValue(g, valueResource, value);
				}
				return true;
			}
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
		
	}

	private Object getValue(ReadGraph g, Resource valueResource)
			throws DoesNotContainValueException, BindingException, ServiceException {
		if (valueBinding != null) {
			return g.getValue(valueResource, getBaseBinding(valueBinding));
		}
		else {
			return g.getValue(valueResource);
		}
	}
	
	private void claimValue(WriteGraph g, Resource valueResource, Object value) throws ServiceException {
		if (valueBinding != null)
			g.claimValue(valueResource, value, getBaseBinding(valueBinding));
		else
			g.claimValue(valueResource, value);
	}

	private static Binding getBaseBinding(Binding binding) {
		return binding instanceof OptionalBinding ? ((OptionalBinding)binding).getComponentBinding() : binding;
	}
	
	private boolean equals(Object o1, Object o2) {
		if (valueBinding != null)
			return valueBinding.equals(o1, o2);
		
		if (o1 instanceof boolean[])
			Arrays.equals((boolean[])o1,(boolean[])o2);
		if (o1 instanceof int[])
			Arrays.equals((int[])o1,(int[])o2);
		if (o1 instanceof float[])
			Arrays.equals((float[])o1,(float[])o2);
		if (o1 instanceof double[])
			Arrays.equals((double[])o1,(double[])o2);
		if (o1 instanceof byte[])
			Arrays.equals((byte[])o1,(byte[])o2);
		return o1.equals(o2);
	}

}
