/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart;

import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.renderer.category.CategoryItemRenderer;
import org.jfree.chart.ui.RectangleInsets;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.sysdyn.JFreeChartResource;

/**
 * Class representing a CategoryPlot for JFreeChart
 * @author Teemu Lempinen
 *
 */
public class CategoryPlot extends AbstractPlot {

    public CategoryPlot(ReadGraph graph, Resource resource) {
        super(graph, resource);
    }

    @Override
    protected Plot newPlot() {
        return new org.jfree.chart.plot.CategoryPlot(null, null, null, null);
    }

    @Override
    protected void setPlotProperties(PlotProperties properties) {
        if(!(plot instanceof org.jfree.chart.plot.CategoryPlot))
            return;
        
        org.jfree.chart.plot.CategoryPlot cplot = (org.jfree.chart.plot.CategoryPlot) plot;
        /* Support using multiple axis, but prefer using only one domain and 
         * one range axis
         */
        for(int i = 0; i < properties.ranges.size(); i++) {
            cplot.setRangeAxis(i, (ValueAxis)properties.ranges.get(i).getAxis());
        }
        
        for(int i = 0; i < properties.domains.size(); i++) {
            cplot.setDomainAxis(i, (CategoryAxis)properties.domains.get(i).getAxis());
        }
       
        
        
        IAxis axis;
        for(int i = 0; i < properties.datasets.size(); i++) {
            IDataset dataset = properties.datasets.get(i);
            org.jfree.data.category.CategoryDataset ds = (org.jfree.data.category.CategoryDataset)dataset.getDataset();
            cplot.setDataset(i, ds);
//            System.out.println("setting dataset " + i + ": " + ds);
            cplot.setRenderer(i, (CategoryItemRenderer)dataset.getRenderer());
            axis = properties.rangeMappings.get(dataset);
            if(axis != null && properties.ranges.contains(axis))
                cplot.mapDatasetToRangeAxis(i, properties.ranges.indexOf(axis));
            axis = properties.domainMappings.get(dataset);
            if(axis != null && properties.ranges.contains(axis))
                cplot.mapDatasetToDomainAxis(i, properties.domains.indexOf(axis));
            
            if (ds instanceof FilteredDataset) {
            	FilteredDataset f = (FilteredDataset)ds;
            	Boolean useFilter = (Boolean)properties.otherProperties.get("useFilter");
                Double filterFraction = (Double)properties.otherProperties.get("filterFraction");
                if (useFilter != null && filterFraction != null) {
                	f.setFiltering(useFilter);
                	f.setFilterFraction(filterFraction*0.01);
                	f.updateFiltered();
                } else {
                	f.setFiltering(false);
                }
            }
        }

        Boolean visibleGrid = (Boolean)properties.otherProperties.get("visibleGrid");
        if(visibleGrid != null) {
            cplot.setRangeGridlinesVisible(visibleGrid);
            cplot.setDomainGridlinesVisible(false);
        }
        
        Boolean orientation = (Boolean)properties.otherProperties.get("orientation");
        if(orientation != null) {
        	if (orientation)
        		cplot.setOrientation(PlotOrientation.HORIZONTAL);
        	else
        		cplot.setOrientation(PlotOrientation.VERTICAL);
        }
        
        // Cleaner look: no outline borders
        cplot.setInsets(new RectangleInsets(2,5,2,2), false);
        cplot.setOutlineVisible(false);
        super.setPlotProperties(properties);
    }

    @Override
    protected void getOtherProperties(ReadGraph graph, PlotProperties properties) throws DatabaseException {
        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
        Boolean visibleGrid = graph.getPossibleRelatedValue(resource, jfree.Plot_visibleGrid);
        properties.otherProperties.put("visibleGrid", visibleGrid);
        Boolean orientation = graph.getPossibleRelatedValue(resource, jfree.Plot_orientation);
        properties.otherProperties.put("orientation", orientation);
        
        Boolean useFilter = graph.getPossibleRelatedValue(resource, JFreeChartResource.getInstance(graph).Filter_used, Bindings.BOOLEAN);
        Double  filterFraction = graph.getPossibleRelatedValue(resource, JFreeChartResource.getInstance(graph).Filter_fraction, Bindings.DOUBLE);
        properties.otherProperties.put("useFilter", useFilter);
        properties.otherProperties.put("filterFraction", filterFraction);
    }

}
