package org.simantics.issues.ui.handler;

import java.util.ArrayList;
import java.util.List;

import org.simantics.Simantics;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.Function;

/**
 * @author Tuukka Lehtonen
 * @since 1.31.0
 */
class FunctionHandler extends PreferenceHandler {

    private String functionPropertyName;
    private Object argument;

    public FunctionHandler(String virtualGraphId, String functionPropertyName, Object argument) {
        super(virtualGraphId, null, false);
        this.functionPropertyName = functionPropertyName;
        this.argument = argument;
    }

    public void execute(List<Variable> vars, List<Resource> resources) {
        if (vars.isEmpty())
            return;
        Session session = Simantics.peekSession();
        if (session == null)
            return;

        VirtualGraph vg = virtualGraphId == null ? null :
                session.getService(VirtualGraphSupport.class).getWorkspacePersistent(virtualGraphId);

        session.asyncRequest(new WriteRequest(vg) {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                graph.markUndoPoint();
                execute(graph, vars, functionPropertyName);
            }
        }, e -> {
            if (e != null)
                Logger.defaultLogError(e);
        });
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    protected void execute(WriteGraph graph, List<Variable> vars, String functionPropertyName) throws DatabaseException {
        List<Function> hiders = new ArrayList<>(vars.size());
        for (Variable v : vars) {
            Function f = v.getPossiblePropertyValue(graph, functionPropertyName);
            if (f != null)
                hiders.add(f);
        }
        if (!hiders.isEmpty()) {
            SCLContext ctx = SCLContext.getCurrent();
            Object oldGraph = ctx.put("graph", graph); //$NON-NLS-1$
            try {
                for (Function f : hiders)
                    f.apply(argument);
            } catch (Throwable t) {
                throw new DatabaseException(t);
            } finally {
                ctx.put("graph", oldGraph); //$NON-NLS-1$
            }
        }
    }

}