/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.graphviz;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

/**
 * Root graph that contains nodes, edges and subgraphs.
 * 
 * @author Hannu Niemist�
 */
public class Graph extends AbstractAttributeContainer implements IGraph {
    Collection<IGraphPart> parts = new ArrayList<IGraphPart>();
    int curId = 0;
    
    public Graph() {
        super();
    }

    public Graph(String label) {
        super();
        setLabel(label);
    }

    @Override
    public void addPart(IGraphPart part) {
        parts.add(part);        
    }

    @Override
    public String newId() {
        return "n" + (++curId);
    }

    @Override
    public Collection<IGraphPart> getParts() {
        return Collections.unmodifiableCollection(parts);
    }

    public void write(PrintStream s) {
        s.print("digraph {");
        s.print("graph");
        writeAttributes(s);
        for(IGraphPart part : parts)
            part.write(s);
        s.println("}");
    }
    
    @Override
    public String toString() {
        ByteArrayOutputStream ba = new ByteArrayOutputStream();
        write(new PrintStream(ba));
        return ba.toString();
    }
    
    public void setLabel(String label) {
        set("label", label);
    }
    
    /**
     * Color used for text.
     */
    public void setFontColor(String color) {
        set("fontcolor", color);
    }
    
    /**
     * <p>Sets direction of graph layout. For example, if rankdir="LR", and barring cycles, an edge T -> H; will go from left to right. By default, graphs are laid out from top to bottom.</p>
     * <p>Alternatives: <tt>TB LR BT RL</tt></p>
     */
    public void setRankdir(String rankdir) {
        set("rankdir", rankdir);
    }
    
    /**
     * <p>In dot, this gives the desired rank separation, in inches. This is the minimum vertical distance between the bottom of the nodes in one rank and the tops of nodes in the next. If the value contains "equally", the centers of all ranks are spaced equally apart. Note that both settings are possible, e.g., ranksep = "1.2 equally". In twopi, specifies radial separation of concentric circles.</p>
     */
    public void setRanksep(double ranksep, boolean equally) {
        set("ranksep", Double.toString(ranksep) + (equally ? " equally" : ""));
    }

}
