/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.element.handler.impl;

import java.awt.BasicStroke;
import java.awt.Stroke;

import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.EdgeVisuals;
import org.simantics.g2d.element.handler.LifeCycle;

/**
 * This implementation is for edges whose visual properties can be modified.
 * 
 * @author Toni Kalajainen
 */
public class ConfigurableEdgeVisuals implements EdgeVisuals, LifeCycle {

    private static final long serialVersionUID = -8662807810086649742L;

    public static final Stroke DEFAULT_STROKE =
        new BasicStroke(0.15f,
                BasicStroke.CAP_BUTT,
                BasicStroke.JOIN_ROUND,
                10.0f, null, 0.0f);

    public static final ConfigurableEdgeVisuals DEFAULT =
        new ConfigurableEdgeVisuals(ArrowType.None, ArrowType.Fill, DEFAULT_STROKE, 1.0, 1.0);

    ArrowType beginArrow, endArrow;
    public Stroke stroke;
    public StrokeType strokeType;
    double beginArrowSize, endArrowSize;

    public ConfigurableEdgeVisuals()
    {
        this.beginArrow = ArrowType.Stroke;
        this.endArrow = ArrowType.Stroke;
        this.stroke = DEFAULT_STROKE;
        this.strokeType = StrokeType.Relative;
        beginArrowSize = 1;
        endArrowSize = 1;
    }

    public ConfigurableEdgeVisuals(ArrowType initialBeginArrow, ArrowType initialEndArrow, Stroke initialStroke, double beginArrowSize, double endArrowSize)
    {
        this.beginArrow = initialBeginArrow;
        this.endArrow = initialEndArrow;
        this.stroke = initialStroke;
        this.strokeType = StrokeType.Relative;
        this.beginArrowSize = beginArrowSize;
        this.endArrowSize = endArrowSize;
    }

    public ConfigurableEdgeVisuals(ArrowType initialBeginArrow, ArrowType initialEndArrow, Stroke initialStroke, StrokeType initialStrokeType, double beginArrowSize, double endArrowSize)
    {
        this.beginArrow = initialBeginArrow;
        this.endArrow = initialEndArrow;
        this.stroke = initialStroke;
        this.strokeType = initialStrokeType;
        this.beginArrowSize = beginArrowSize;
        this.endArrowSize = endArrowSize;
    }

    @Override
    public ArrowType getArrowType(IElement e, EdgeEnd end) {
        if (end == EdgeEnd.Begin)
            return e.getHint(ElementHints.KEY_BEGIN_ARROW);
        if (end == EdgeEnd.End)
            return e.getHint(ElementHints.KEY_END_ARROW);
        return null;
    }

    @Override
    public StrokeType getStrokeType(IElement e) {
        return e.getHint(ElementHints.KEY_EDGE_STROKE_TYPE);
    }

    @Override
    public Stroke getStroke(IElement e) {
        return e.getHint(ElementHints.KEY_EDGE_STROKE);
    }

    @Override
    public void setArrowType(IElement e, EdgeEnd end, ArrowType arrowType) {
        if (end == EdgeEnd.Begin)
            e.setHint(ElementHints.KEY_BEGIN_ARROW, arrowType);
        if (end == EdgeEnd.End)
            e.setHint(ElementHints.KEY_END_ARROW, arrowType);
    }

    @Override
    public void setStrokeType(IElement e, StrokeType strokeType) {
        e.setHint(ElementHints.KEY_EDGE_STROKE_TYPE, strokeType);
    }

    @Override
    public void setStroke(IElement e, Stroke s) {
        e.setHint(ElementHints.KEY_EDGE_STROKE, s);
    }

    @Override
    public void onElementActivated(IDiagram d, IElement e) {
    }

    @Override
    public void onElementCreated(IElement e) {
        e.setHint(ElementHints.KEY_BEGIN_ARROW, beginArrow);
        e.setHint(ElementHints.KEY_END_ARROW, endArrow);
        e.setHint(ElementHints.KEY_EDGE_STROKE, stroke);
        e.setHint(ElementHints.KEY_EDGE_STROKE_TYPE, strokeType);
        e.setHint(ElementHints.KEY_BEGIN_ARROW_SIZE, beginArrowSize);
        e.setHint(ElementHints.KEY_END_ARROW_SIZE, endArrowSize);
    }

    @Override
    public void onElementDeactivated(IDiagram d, IElement e) {
    }

    @Override
    public void onElementDestroyed(IElement e) {
    }

    @Override
    public double getArrowSize(IElement e, EdgeEnd end) {
        if (end==EdgeEnd.Begin)
            return e.getHint(ElementHints.KEY_BEGIN_ARROW_SIZE);
        else
            if (end==EdgeEnd.End)
                return e.getHint(ElementHints.KEY_END_ARROW_SIZE);
        return 0;
    }

    @Override
    public void setArrowSize(IElement e, EdgeEnd end, double size) {
        if (end==EdgeEnd.Begin)
            e.setHint(ElementHints.KEY_BEGIN_ARROW_SIZE, size);
        else
            if (end==EdgeEnd.End)
                e.setHint(ElementHints.KEY_END_ARROW_SIZE, size);
    }

}
