/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.diagram;

import java.util.Comparator;
import java.util.List;

import org.simantics.g2d.diagram.impl.Diagram;
import org.simantics.g2d.diagram.participant.ZOrderHandler;
import org.simantics.g2d.element.IElement;
import org.simantics.utils.datastructures.hints.IHintContext;

/**
 * Diagram is a flat ordered composition of elements.
 * <p>
 * Diagram may only be accessed from canvas thread.
 * <p>
 * Diagram variables are persistent.
 * <p>
 * Elements are ordered according to their z-value
 * 
 * @see Diagram Default implementation
 * @see IElement
 * @see ZOrderHandler
 * @author Toni Kalajainen
 */
public interface IDiagram extends IHintContext {

    public interface CompositionVetoListener {
        /**
         * Invoked before an element is actually added on to a diagram. This
         * provides the possibility to veto the addition by returning
         * <code>false</code>.
         * 
         * @param d the diagram to be changed
         * @param e the element to be added
         * @return <code>true</code> to allow the addition or <code>false</code>
         *         to veto the addition
         */
        boolean beforeElementAdded(IDiagram d, IElement e);

        /**
         * Invoked before an element is actually removed from a diagram. This
         * provides the possibility to veto the addition by returning
         * <code>false</code>.
         * 
         * @param d the diagram to be changed
         * @param e the element to be removed
         * @return <code>true</code> to allow the removal or <code>false</code>
         *         to veto the removal
         */
        boolean beforeElementRemoved(IDiagram d, IElement e);
    }

    public interface CompositionListener {
        /**
         * @param d
         * @param e
         */
        void onElementAdded(IDiagram d, IElement e);

        /**
         * @param d
         * @param e
         */
        void onElementRemoved(IDiagram d, IElement e);
    }

    /**
     * Add a composition listener that can veto element addition/removal
     * @param listener
     */
    void addCompositionVetoListener(CompositionVetoListener listener);
    /**
     * Remove composition veto listener
     * @param listener
     */
    void removeCompositionVetoListener(CompositionVetoListener listener);

    /**
     * Add a composition listener that notifies about element composition
     * @param listener
     */
    void addCompositionListener(CompositionListener listener);
    /**
     * Remove composition listener
     * @param listener
     */
    void removeCompositionListener(CompositionListener listener);

    /**
     * Get diagram class
     * @return diagram class
     */
    DiagramClass getDiagramClass();

    /**
     * Get a snapshot of elements. The list is z-ordered, where
     * top most elements are at the end of the list.
     * @return a snapshot of elements
     */
    List<IElement> getSnapshot();

    /**
     * Get z-ordered list of the elements. The return value is not intended for
     * direct modification.
     * 
     * @return
     */
    List<IElement> getElements();

    /**
     * Reorder the contained diagram elements using the specified comparator.
     * 
     * @param comparator
     */
    void sort(Comparator<IElement> comparator);

    /**
     * Add an element to the diagram.
     * 
     * @param element the element to add
     * @return element of class
     */
    void addElement(IElement element);

    /**
     * Checks if a diagram contains an element.
     * @return true if diagram contains the element.
     */
    boolean containsElement(IElement element);

    /**
     * Remove an element from the diagram
     * @param element element to remove
     */
    void removeElement(IElement element);

    /**
     * Destroy the diagram from the world permanently.
     * Destroys all contained elements as well.
     */
    void destroy();
    void dispose();

    // Z-Order
    boolean bringUp(IElement e);
    boolean sendDown(IElement e);
    boolean bringToTop(IElement e);
    boolean sendToBottom(IElement e);

    /**
     * @param e
     * @param position
     * @return <code>true</code> if the position changed
     */
    boolean moveTo(IElement e, int position);

}
