/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.synchronization.graph;

import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.content.ConnectionUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.CopyAdvisor;
import org.simantics.diagram.synchronization.IModifiableSynchronizationContext;
import org.simantics.diagram.synchronization.ModificationAdapter;
import org.simantics.diagram.synchronization.SynchronizationHints;
import org.simantics.diagram.synchronization.graph.layer.GraphLayerManager;
import org.simantics.diagram.ui.DiagramModelHints;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.DiagramMutator;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.layer0.Layer0;
import org.simantics.structural.stubs.StructuralResource2;

/**
 * @author Tuukka Lehtonen
 */
public class AddConnection extends ModificationAdapter {

    IModifiableSynchronizationContext context;
    IDiagram                          diagram;
    Resource                          diagramResource;
    IElement                          element;
    Resource                          copyOf;
    Resource                          connectionClassResource;

    public AddConnection(IModifiableSynchronizationContext context, IDiagram diagram, IElement element) {
        super(ADD_CONNECTION_PRIORITY);
        this.context = context;
        this.diagram = diagram;
        this.diagramResource = diagram.getHint(DiagramModelHints.KEY_DIAGRAM_RESOURCE);
        assert this.diagramResource != null;
        this.element = element;
        this.copyOf = element.getHint(ElementHints.KEY_COPY_OF_OBJECT);

        this.connectionClassResource = ElementUtils.checkedAdapt(element.getElementClass(), Resource.class);
    }

    @Override
    public void perform(WriteGraph graph) throws DatabaseException {
        ConnectionUtil cu = new ConnectionUtil(graph);
        DiagramMutator mutator = diagram.getHint(DiagramHints.KEY_MUTATOR);

        Resource resource = null;

        // 3. Try to copy the element from an existing element if requested.
        if (copyOf != null) {
            CopyAdvisor ca = diagram.getHint(SynchronizationHints.COPY_ADVISOR);
            if (ca != null) {
                Resource sourceDiagram = graph.getPossibleObject(copyOf, Layer0.getInstance(graph).PartOf);
                if (sourceDiagram == null)
                    sourceDiagram = OrderedSetUtils.getSingleOwnerList(graph, copyOf, DiagramResource.getInstance(graph).Diagram);
                resource = CopyAdvisorUtil.copy(context, graph, ca, copyOf, sourceDiagram, diagramResource);
                if (resource != null) {
                    OrderedSetUtils.addFirst(graph, diagramResource, resource);
                }
            }
        }

        if (resource == null)
            resource = cu.newConnection(diagramResource, connectionClassResource);

        if (copyOf != null)
            copyConnectionType(graph, copyOf, resource);

        element.setHint(ElementHints.KEY_OBJECT, resource);
        mutator.register(element, resource);

        GraphLayerManager glm = context.get(GraphSynchronizationHints.GRAPH_LAYER_MANAGER);
        if (glm != null) {
            glm.removeFromAllLayers(graph, resource);
            glm.putElementOnVisibleLayers(diagram, graph, resource);
        }
    }

    public static void copyConnectionType(WriteGraph graph, Resource source, Resource target) throws DatabaseException {
        // Copy HasConnectionType relation
        StructuralResource2 sr = StructuralResource2.getInstance(graph);
        Resource connectionType = graph.getPossibleObject(source, sr.HasConnectionType);
        if (connectionType != null) {
            graph.deny(target, sr.HasConnectionType);
            graph.claim(target, sr.HasConnectionType, null, connectionType);
        }
    }

}
