/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - index based searching and graph manipulation (#4255)
 *******************************************************************************/
package org.simantics.debug.ui;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;
import org.simantics.Simantics;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.debug.ui.internal.Activator;
import org.simantics.debug.ui.internal.DebugUtils;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.workbench.ResourceInput;
import org.simantics.utils.ui.BundleUtils;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.LayoutUtils;

public class GraphDebuggerView extends ViewPart {

    public static final String VIEW_ID              = "org.simantics.debug.graphDebugger"; //$NON-NLS-1$

//    private final boolean      DEFAULT_RECYCLE_VIEW = true;

    private ResourceInput      input;

    private Session            session;

    private GraphDebugger      debugger;

//    private IAction            recycleToggle;
    private IAction            backAction;
    private IAction            forwardAction;
    private IAction            refreshAction;
    private IAction            findAction;
    private IAction            addStatementAction;
    private IAction            addResourceAction;
    private IAction            homeAction;

    @Override
    public void createPartControl(Composite parent) {
//      System.out.println("koss: " + getViewSite().getSecondaryId());
        session = Simantics.getSession();

        // Initialize input
        String sid = getViewSite().getSecondaryId();
        Resource r = null;
        if (sid != null) {
            input = ResourceInput.unmarshall(sid);
            try {
                r = input.toResource(session);
            } catch (DatabaseException e) {
                ErrorLogger.defaultLogError(e);
            }
        }

        // Create UI
        parent.setLayout(LayoutUtils.createNoBorderGridLayout(1));
        debugger = new GraphDebugger(parent, SWT.NONE, session, r, getSite());
        debugger.defaultInitializeUI();

        // Contribute actions
        makeActions();
        contributeActions(getViewSite().getActionBars());
        updateActionStates();
    }

    @Override
    public void dispose() {
        super.dispose();
    }

    private void makeActions() {
//        recycleToggle = new RecycleToggle();
//        recycleToggle.setChecked(DEFAULT_RECYCLE_VIEW);

        backAction = new BackAction();
        forwardAction = new ForwardAction();
        refreshAction = new RefreshAction();
        findAction = new FindAction();
        addStatementAction = new AddStatementAction();
        addResourceAction = new AddResourceAction();
        homeAction = new HomeAction();
    }

    private void contributeActions(IActionBars actionBars) {
        IToolBarManager toolBar = actionBars.getToolBarManager();
        toolBar.add(backAction);
        toolBar.add(forwardAction);
        toolBar.add(new Separator());
        toolBar.add(refreshAction);
        toolBar.add(new Separator());
        toolBar.add(findAction);
        toolBar.add(addStatementAction);
        toolBar.add(addResourceAction);
        toolBar.add(new Separator());
        toolBar.add(homeAction);

//        toolBar.add(recycleToggle);
    }

    @Override
    public void setFocus() {
        if (debugger != null)
            debugger.setFocus();
    }

    private void refreshBrowser() {
        debugger.refreshBrowser();
    }

//    class RecycleToggle extends Action {
//        public RecycleToggle() {
//            super("Open New Views", Action.AS_CHECK_BOX);
//            setImageDescriptor(Activator.getImageDescriptor("icons/reload.gif"));
//            setToolTipText("Toggles whether or not to open new views when resource links are clicked.");
//        }
//    }
//
    class RefreshAction extends Action {
        public RefreshAction() {
            super(Messages.GraphDebuggerView_Refresh, BundleUtils.getImageDescriptorFromPlugin(SimanticsUI.PLUGIN_ID, "icons/refresh.gif")); //$NON-NLS-2$
            setToolTipText(Messages.GraphDebuggerView_Refresh);
        }
        @Override
        public void run() {
            refreshBrowser();
        }
    }

    class BackAction extends Action {
        public BackAction() {
            super(Messages.GraphDebuggerView_Back, Action.AS_PUSH_BUTTON);
            setToolTipText(Messages.GraphDebuggerView_BackTT);
            setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_BACK));
            setDisabledImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_BACK_DISABLED));
        }
        @Override
        public void run() {
            back();
        }
    }

    class ForwardAction extends Action {
        public ForwardAction() {
            super(Messages.GraphDebuggerView_Forward, Action.AS_PUSH_BUTTON);
            setToolTipText(Messages.GraphDebuggerView_ForwardTT);
            setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_FORWARD));
            setDisabledImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_TOOL_FORWARD_DISABLED));
        }
        @Override
        public void run() {
            forward();
        }
    }

    class HomeAction extends Action {
        public HomeAction() {
            super(Messages.GraphDebuggerView_Home, Action.AS_PUSH_BUTTON);
            setToolTipText(Messages.GraphDebuggerView_HomeTT);
            setImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_ETOOL_HOME_NAV));
            setDisabledImageDescriptor(PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_ETOOL_HOME_NAV_DISABLED));
        }
        @Override
        public void run() {
            navigateHome();
        }
    }

    class FindAction extends Action {
        public FindAction() {
            super(Messages.GraphDebuggerView_Find, BundleUtils.getImageDescriptorFromPlugin(Activator.PLUGIN_ID, "icons/cog_blue.png")); //$NON-NLS-2$
            setToolTipText(Messages.GraphDebuggerView_FindTT);
        }
        @Override
        public void run() {
            DebugUtils.find(Simantics.getSession(), debugger);
        }
    }

    class AddStatementAction extends Action {
        public AddStatementAction() {
            super(Messages.GraphDebuggerView_AddStatement, BundleUtils.getImageDescriptorFromPlugin(Activator.PLUGIN_ID, "icons/cog_add.png")); //$NON-NLS-2$
            setToolTipText(Messages.GraphDebuggerView_AddStatementTT);
        }
        @Override
        public void run() {
            try {
                DebugUtils.addStatement(Simantics.getSession(), debugger);
            } catch (DatabaseException e) {
                Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, e.getMessage(), e));
            }
        }
    }

    class AddResourceAction extends Action {
        public AddResourceAction() {
            super(Messages.GraphDebuggerView_AddResource, BundleUtils.getImageDescriptorFromPlugin(Activator.PLUGIN_ID, "icons/cog_add.png")); //$NON-NLS-2$
            setToolTipText(Messages.GraphDebuggerView_AddResourceTT);
        }
        @Override
        public void run() {
            try {
                DebugUtils.addResource(Simantics.getSession(), debugger);
            } catch (DatabaseException e) {
                Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, e.getMessage(), e));
            }
        }
    }

    private void back() {
        debugger.back();
    }

    private void forward() {
        debugger.forward();
    }

    private void navigateHome() {
        Resource rootLibrary = session.getRootLibrary();
        debugger.changeLocation(rootLibrary);
    }

    private void updateActionStates() {
        backAction.setEnabled(debugger.hasBackHistory());
        forwardAction.setEnabled(debugger.hasForwardHistory());
    }

}
