/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.simantics.databoard.accessor.Accessor;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.util.binary.RandomAccessBinary;
import org.simantics.db.adaption.Adapter;
import org.simantics.db.exception.AdaptionException;
import org.simantics.db.exception.AssumptionException;
import org.simantics.db.exception.BindingException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.DoesNotContainValueException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.exception.NoInverseException;
import org.simantics.db.exception.NoSingleResultException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.procedure.SyncListener;
import org.simantics.db.procedure.SyncMultiListener;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.procedure.SyncProcedure;
import org.simantics.db.procedure.SyncSetListener;
import org.simantics.db.request.Read;
import org.simantics.scl.compiler.types.Type;

/**
 * 
 * For accessing and manipulating the graph data in synchronous manner. A client
 * receives ReadGraph instances for performing requests in {@link Read} and
 * {@link MultiRead} as well as for reacting to request results using
 * {@link SyncProcedure}, {@link SyncListener}, {@link SyncMultiProcedure},
 * {@link SyncMultiListener} and {@link SyncSetListener}
 * <p>
 * ReadGraph instances are stateful and may not be stored and used outside the
 * supplier method activation. The methods receiving ReadGraph should not block
 * for long periods of time since that may delay the processing of other
 * requests. For an asynchronous counterpart with the same functionality as
 * ReadGraph see {@link AsyncReadGraph}
 * 
 * <p>
 * All collections returned by the methods on this interface must not be
 * modified by clients unless explicitly stated otherwise.
 * 
 * @version 0.7
 * @author Antti Villberg
 * @see RequestProcessor
 * @see AsyncRequestProcessor
 * @see AsyncReadGraph
 * @see Resource
 * @see Statement
 * @noimplement
 */
public interface ReadGraph extends RequestProcessor {

    public static final String GRAPH_HINT_SYNCHRONOUS = "sync";
    
    /**
     * 
     * @see AsyncReadGraph#forURI
     */
    String getURI(Resource resource) throws AssumptionException, ValidationException, ServiceException;
    String getPossibleURI(Resource resource) throws ValidationException, ServiceException;

    /**
     * Gets a unique resource associated to the given identifier. See <a
     * href="https://www.simantics.org/wiki/index.php/URI">Simantics Wiki page
     * on URIs</a> for more details.
     * 
     * @param uri the identifier
     * @return the resource
     * @throws ResourceNotFoundException if a resource for the given identifier
     *         was not specified
     * @throws ValidationException if a resource could not be produced due to
     *         invalid semantics
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forResourceByURI
     */
    Resource getResource(String uri) throws ResourceNotFoundException, ValidationException, ServiceException;

    /**
     * Gets a unique resource associated to the given identifier. See <a
     * href="https://www.simantics.org/wiki/index.php/URI">Simantics Wiki page
     * on URIs</a> for more details.
     * 
     * @param uri the identifier
     * @return the resource
     * @throws ResourceNotFoundException if a resource for the given identifier
     *         was not specified
     * @throws ValidationException if a resource could not be produced due to
     *         invalid semantics
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forResourceByURI
     */
    Resource getPossibleResource(String uri) throws ResourceNotFoundException, ValidationException, ServiceException;

    /**
     * Computes a map of objects related to resource with L0.ConsistsOf that also contain a L0.HasName property
     * 
     * @param resource the resource
     * @return the children
     * @throws ValidationException if a resource could not be produced due to
     *         invalid semantics
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forResourceByURI
     */
    Map<String,Resource> getChildren(Resource resource) throws ValidationException, ServiceException;

    /**
     * Gets a builtin resource. For a list of builtin resources see TODO Wiki
     * 
     * @param uri the identifier
     * @return the resource
     * @throws ResourceNotFoundException if resouce was not found
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forBuiltin
     */
    Resource getBuiltin(String id) throws ResourceNotFoundException, ServiceException;

    /**
     * Gets all statements (subject, p, o) such that
     * {@link ReadGraph#isSubrelationOf(p, relation)} returns true. The result
     * may include asserted statements for which s does not equal subject. See
     * the assertion mechanisn TODO for more details. If the relation is
     * functional the request returns a single statement according to the
     * functional statement shadowing rules TODO if possible. If an invalid
     * graph is encountered an exception is thrown.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the statements
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forEachStatement
     * @see AsyncReadGraph#forStatementSet
     */
    Collection<Statement> getStatements(Resource subject, Resource relation)
    throws ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * Gets all statements (s, p, o) asserted by subject such that
     * {@link ReadGraph#isSubrelationOf(p, relation)} returns true. See the
     * assertion mechanism TODO for more details. If the relation is functional
     * the request returns a single statement according to the functional
     * statement shadowing rules TODO if possible. If an invalid graph is
     * encountered an exception is thrown.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the statements
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forEachAssertedStatement
     * @see AsyncReadGraph#forAssertedStatementSet
     */
    Collection<Statement> getAssertedStatements(Resource subject, Resource relation)
    throws ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * Gets all predicates p from statements (subject, p, o) such that
     * {@link ReadGraph#isSubrelationOf(p, relation)} returns true. See the
     * assertion mechanism TODO for more details. If the relation is functional
     * the request returns a single statement according to the functional
     * statement shadowing rules TODO if possible. If an invalid graph is
     * encountered an exception is thrown.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the predicates
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forEachPredicate
     * @see AsyncReadGraph#forPredicateSet
     */
    Collection<Resource> getPredicates(Resource subject) throws ServiceException;

    /**
     * Gets a subset of all types of the subject such that no pair of returned
     * types inherit each other and all types of the subject are supertypes
     * of some type in the returned set.
     * 
     * @param subject a resource
     * @return the principal types
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forEachPrincipalType
     * @see AsyncReadGraph#forPrincipalTypeSet
     */
    Collection<Resource> getPrincipalTypes(Resource subject) throws ServiceException;

    /**
     * Gets the full type hierarchy for subject.
     * 
     * @param subject a resource
     * @return the type hierarchy
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forTypes
     */
    Set<Resource> getTypes(Resource subject) throws ServiceException;

    /**
     * Gets the full supertype hierarchy for subject.
     * 
     * @param subject a resource
     * @return the supertype hierarchy
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forSupertypes
     */
    Set<Resource> getSupertypes(Resource subject) throws ServiceException;

    /**
     * Gets the full superrelation hierarchy for subject.
     * 
     * @param subject a resource
     * @return the superrelation hierarchy
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forSuperrelations
     */
    Set<Resource> getSuperrelations(Resource subject) throws ServiceException;
    Resource getPossibleSuperrelation(Resource subject) throws ServiceException;

    /**
     * Gets all objects o from statements (subject, p, o) such that
     * {@link ReadGraph#isSubrelationOf(p, relation)} returns true. See the
     * assertion mechanisn TODO for more details. If the relation is functional
     * the request returns a single object according to the functional statement
     * shadowing rules TODO if possible. If an invalid graph is encountered an
     * exception is thrown.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the objects
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forEachObject
     * @see AsyncReadGraph#forObjectSet
     */
    Collection<Resource> getObjects(Resource subject, Resource relation)
    throws ServiceException;

    /**
     * Gets all objects o (s, p, o) asserted by subject such that
     * {@link ReadGraph#isSubrelationOf(p, relation)} returns true. See the
     * assertion mechanism TODO for more details. If the relation is functional
     * the request returns a single object according to the functional statement
     * shadowing rules TODO if possible. If an invalid graph is encountered an
     * exception is thrown.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the objects
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forEachAssertedStatement
     * @see AsyncReadGraph#forAssertedStatementSet
     */
    Collection<Resource> getAssertedObjects(Resource subject, Resource relation)
    throws ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * Gets the inverse r of relation such that (relation, InverseOf, r). The
     * methods assumes an inverse exists.
     * 
     * @param relation a resource
     * @return the inverse
     * @throws NoInverseException if inverse was not fond
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forInverse
     */
    Resource getInverse(Resource relation) throws NoInverseException, ManyObjectsForFunctionalRelationException,
    ServiceException;

    /**
     * Gets a single object as returned by {@link ReadGraph#getObjects}. Assumes
     * a single object.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the single object
     * @throws NoSingleResultException if the amount of valid objects was not
     *         one
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forSingleObject
     */
    Resource getSingleObject(Resource subject, Resource relation) throws NoSingleResultException,
    ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * Gets a single statement as returned by {@link ReadGraph#getStatements}.
     * Assumes a single statement.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the single statement
     * @throws NoSingleResultException if the amount of valid statements was not
     *         one
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forSingleStatement
     */
    Statement getSingleStatement(Resource subject, Resource relation) throws NoSingleResultException,
    ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * Gets a single type t as returned by {@link ReadGraph#getPrincipalTypes}. Assumes a single type.
     * 
     * @param subject a resource
     * @param baseType a resource
     * @return the single type
     * @throws NoSingleResultException if the amount of valid types was not one
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forSingleType
     */
    Resource getSingleType(Resource subject) throws NoSingleResultException, ServiceException;

    /**
     * Gets a single type t as returned by {@link ReadGraph#getTypes} where
     * {@link ReadGraph#isInheritedFrom(t, baseType)}. Assumes a single type.
     * 
     * @param subject a resource
     * @param baseType a resource
     * @return the single type
     * @throws NoSingleResultException if the amount of valid types was not one
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forSingleType
     */
    Resource getSingleType(Resource subject, Resource baseType) throws NoSingleResultException, ServiceException;

    /**
     * Gets a value associated to subject. Infers {@link Binding} from type t =
     * {@link ReadGraph#getSingleType(subject, Value)}
     * 
     * @param subject a resource
     * @return the value
     * @throws DoesNotContainValueException if the subject did not contain a
     *         resource
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forValue
     */
    <T> T getValue(Resource subject) throws DoesNotContainValueException, ServiceException;

    /**
     * Gets a value associated to subject as a Variant. Infers {@link Binding} from type t =
     * {@link ReadGraph#getSingleType(subject, Value)}
     * 
     * @param subject a resource
     * @return the value
     * @throws DoesNotContainValueException if the subject did not contain a
     *         resource
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forValue
     */
    Variant getVariantValue(Resource subject) throws DoesNotContainValueException, ServiceException;
    
    /**
     * Gets a value associated to subject using the given {@link Binding}.
     * 
     * @param subject a resource
     * @param binding a binding
     * @return the value
     * @throws DoesNotContainValueException if the subject did not contain a
     *         resource
     * @throws BindingException if value could not be produced using the given
     *         {@link Binding}
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forValue
     */
    <T> T getValue(Resource subject, Binding binding) throws DoesNotContainValueException, BindingException,
    ServiceException;

    /**
     * Gets a single value associated to o such that (subject, r, o) and
     * {@link ReadGraph#isSubrelationOf(r, relation)}. Assumes a single value.
     * Infers {@link Binding} from type t =
     * {@link ReadGraph#getSingleType(o, Value)}
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the value
     * @throws NoSingleResultException if the amount of valid values was not one
     * @throws DoesNotContainValueException if suitable o did not contain a
     *         resource
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forRelatedValue
     */
    <T> T getRelatedValue(Resource subject, Resource relation) throws NoSingleResultException,
    DoesNotContainValueException, ServiceException;

    /**
     * Gets a single value associated to o such that (subject, r, o) and
     * {@link ReadGraph#isSubrelationOf(r, relation)} as a Variant. Assumes a single value.
     * Infers {@link Binding} from type t =
     * {@link ReadGraph#getSingleType(o, Value)}
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the value
     * @throws NoSingleResultException if the amount of valid values was not one
     * @throws DoesNotContainValueException if suitable o did not contain a
     *         resource
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forRelatedValue
     */
    Variant getRelatedVariantValue(Resource subject, Resource relation) throws NoSingleResultException,
    DoesNotContainValueException, ServiceException;    

    /**
     * Gets a single value associated to o such that (subject, r, o) and
     * {@link ReadGraph#isSubrelationOf(r, relation)}. Uses the given
     * {@link Binding}. Assumes a single value.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the value
     * @throws NoSingleResultException if the amount of valid values was not one
     * @throws DoesNotContainValueException if suitable o did not contain a
     *         resource
     * @throws BindingException if value could not be produced using the given
     *         {@link Binding}
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forRelatedValue
     */
    <T> T getRelatedValue(Resource subject, Resource relation, Binding binding) throws NoSingleResultException,
    DoesNotContainValueException, BindingException, ServiceException;

    /**
     * Tries to adapt the specified resource into an instance of the specified
     * class T. No guarantees are made on return values of this method. The
     * returned values can be cached results of previous invocation or they can
     * be new result object instances. See {@link #adaptUnique(Resource, Class)}
     * for alternative behaviour.
     * 
     * @param resource a resource
     * @param clazz the adapter class
     * @return
     * @throws AdaptionException if suitable adapter was not found
     * @throws ValidationException from the adapter
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forAdapted
     * @see #adaptUnique(Resource, Class)
     */
    <T> T adapt(Resource resource, Class<T> clazz) throws AdaptionException, ValidationException, ServiceException;

    /**
     * 
     * Same as: adapt(getSingleObject(resource, relation), clazz);
     * 
     * @param resource a resource
     * @param relation a resource
     * @param clazz the adapter class
     * @return
     * @throws AdaptionException if suitable adapter was not found
     * @throws ValidationException from the adapter
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forAdapted
     * @see #adaptUnique(Resource, Class)
     */
    <T> T adaptRelated(Resource resource, Resource relation, Class<T> clazz) throws AdaptionException, NoSingleResultException, ValidationException, ServiceException;

    /**
     * 
     * Same as:
     * 
     *  Resource r = getPossibleObject(resource, relation);
     *  if(r == null) return null;
     *  return getPossibleAdapter(r, clazz);
     * 
     * @param resource a resource
     * @param relation a resource
     * @param clazz the adapter class
     * @return
     * @throws AdaptionException if suitable adapter was not found
     * @throws ValidationException from the adapter
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forAdapted
     * @see #adaptUnique(Resource, Class)
     */
    <T> T getPossibleRelatedAdapter(Resource resource, Resource relation, Class<T> clazz) throws ValidationException, ServiceException;

    <T,C> T adaptContextual(Resource resource, C context, Class<C> contextClass, Class<T> clazz) throws AdaptionException, NoSingleResultException, ValidationException, ServiceException;
    <T,C> T getPossibleContextualAdapter(Resource resource, C context, Class<C> contextClass, Class<T> clazz) throws ValidationException, ServiceException;
    
    /**
     * Tries to adapt the specified resource into an instance of the specified
     * class T. The difference to {@link #adapt(Resource, Class)} is that this
     * method does everything possible to return a new object instance for each
     * separate invocation.
     * 
     * <p>
     * The only case when this cannot be guaranteed is when the adaption result
     * comes from an {@link Adapter} implementation which can internally do
     * anything, like always return the same singleton instance.
     * </p>
     * 
     * @param resource a resource
     * @param clazz the adapter class
     * @return
     * @throws AdaptionException if suitable adapter was not found
     * @throws ValidationException from the adapter
     * @throws ServiceException on connection and database failure
     * @see AsyncReadGraph#forAdapted
     * @see #adapt(Resource, Class)
     */
    <T> T adaptUnique(Resource resource, Class<T> clazz) throws AdaptionException, ValidationException,
    ServiceException;

    /**
     * As {@link #getInverse} but returns null instead of an exception if
     * inverse was not found.
     * 
     * @param relation a resource
     * @return the inverse or null
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forPossibleInverse
     */
    Resource getPossibleInverse(Resource relation) throws ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * As {@link #getSingleObject} but returns null instead of an exception if
     * object was not found.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the object or null
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forPossibleObject
     */
    Resource getPossibleObject(Resource subject, Resource relation) throws ManyObjectsForFunctionalRelationException,
    ServiceException;

    /**
     * As {@link #getSingleStatement} but returns null instead of an exception
     * if statement was not found.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the statement or null
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forPossibleStatement
     */
    Statement getPossibleStatement(Resource subject, Resource relation)
    throws ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * As {@link #getSingleType} but returns null instead of an exception if
     * type was not found.
     * 
     * @param subject a resource
     * @param baseType a resource
     * @return the type or null
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forPossibleType
     */
    Resource getPossibleType(Resource subject, Resource baseType) throws ServiceException;

    /**
     * As {@link #getValue} but returns null instead of an exception if value
     * was not found.
     * 
     * @param subject a resource
     * @return the value or null
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forPossibleValue
     */
    <T> T getPossibleValue(Resource subject) throws ServiceException;

    /**
     * As {@link #getValue} but returns null instead of an exception if value
     * was not found.
     * 
     * @param subject a resource
     * @param binding a {@link Binding}
     * @return the value or null
     * @throws BindingException if value could not be produced using the given
     *         {@link Binding}
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forPossibleValue
     */
    <T> T getPossibleValue(Resource subject, Binding binding) throws BindingException, ServiceException;

    /**
     * As {@link #getPossibleRelatedValue} but returns null instead of an
     * exception if value was not found.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return the value or null
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forPossibleRelatedValue
     */
    <T> T getPossibleRelatedValue(Resource subject, Resource relation)
    throws ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * As {@link #getPossibleRelatedValue} but returns null instead of an
     * exception if value was not found.
     * 
     * @param subject a resource
     * @param relation a resource
     * @param binding a {@link Binding}
     * @return the value or null
     * @throws ManyObjectsForFunctionalRelationException on invalid graph with
     *         more than one functional object
     * @throws BindingException if value could not be produced using the given
     *         {@link Binding}
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forPossibleRelatedValue
     */
    <T> T getPossibleRelatedValue(Resource subject, Resource relation, Binding binding)
    throws ManyObjectsForFunctionalRelationException, BindingException, ServiceException;

    /**
     * As {@link #adapt} but returns null instead of an exception if adapter was
     * not found
     * 
     * @param subject a resource
     * @param clazz a Class
     * @return the adapter or null
     * @throws ValidationException when ?
     * @throws ServiceException on connection and database failures
     * @see {@link AsyncReadGraph#forPossibleAdapted}
     */
    <T> T getPossibleAdapter(Resource resource, Class<T> clazz) throws ValidationException, ServiceException;

    /**
     * As {@link #adaptUnique} but returns null instead of an exception if
     * adapter was not found
     * 
     * @param subject a resource
     * @param clazz a Class
     * @return the adapter or null
     * @throws ValidationException when ?
     * @throws ServiceException on connection and database failures
     * @see {@link AsyncReadGraph#forPossibleAdapted}
     */
    <T> T getPossibleUniqueAdapter(Resource resource, Class<T> clazz) throws ValidationException, ServiceException;

    /**
     * Returns true if type is included in the result of {@link #getTypes(resource)}
     * 
     * @param subject a resource
     * @param type a resource
     * @return indicates if resource is an instance of type
     * @throws ServiceException on connection and database failures
     * @see {@link AsyncReadGraph#forIsInstanceOf}
     */
    boolean isInstanceOf(Resource resource, Resource type) throws ServiceException;

    /**
     * Returns true if type equals resource or type is included in the result of
     * {@link #getSupertypes(resource)}
     * 
     * @param subject a resource
     * @param type a resource
     * @return indicates if resource is inherited from type
     * @throws ServiceException on connection and database failures
     * @see {@link AsyncReadGraph#forIsInheritedFrom}
     */
    boolean isInheritedFrom(Resource resource, Resource type) throws ServiceException;

    /**
     * Returns true if relation equals resource or relation is included in the result of
     * {@link #getSuperrelations(resource)}
     * 
     * @param subject a resource
     * @param relation a resource
     * @return indicates if resource is a subrelation of relation
     * @throws ServiceException on connection and database failures
     * @see {@link AsyncReadGraph#forIsSubrelationOf}
     */
    boolean isSubrelationOf(Resource resource, Resource relation) throws ServiceException;

    /**
     * Returns true if the result of {@link #getStatements(subject, IsWeaklyRelatedTo)} is nonempty.
     * 
     * @param subject a resource
     * @return indicates that subject has statements
     * @throws ServiceException on connection and database failures
     * @see {@link AsyncReadGraph#forHasStatement}
     */
    boolean hasStatement(Resource subject) throws ServiceException;

    /**
     * Returns true if the result of {@link #getStatements(subject, relation)} is nonempty.
     * 
     * @param subject a resource
     * @param relation a resource
     * @return indicates that subject has statements with predicates as subrelation of relation
     * @throws ServiceException on connection and database failures
     * @see {@link AsyncReadGraph#forHasStatement}
     */
    boolean hasStatement(Resource subject, Resource relation) throws ServiceException;

    /**
     * Returns true if a statement (subject, relation, object) exists after assertions.
     * 
     * @param subject a resource
     * @param relation a resource
     * @param object a resource
     * @return indicates that the given statement exists
     * @throws ServiceException on connection and database failures
     * @see {@link AsyncReadGraph#forHasStatement}
     */
    boolean hasStatement(Resource subject, Resource relation, Resource object) throws ServiceException;

    /**
     * Returns true if there is a value associated with subject.
     * IMPLEMENTATION NOTE:
     * Current implementation fetches the value from server. If the value is
     * large (i.e. not stored in the cluster) this is highly inefficient.
     * 
     * @param subject a resource
     * @return indicates that a value exists
     * @throws ServiceException on connection and database failures
     * @see {@link AsyncReadGraph#forHasValue}
     */
    boolean hasValue(Resource subject) throws ServiceException;

    /**
     * Returns the data type of the literal.
     * @param subject
     * @return
     * @throws DatabaseException
     */
    Datatype getDataType(Resource subject) throws DatabaseException;

    /**
     * Returns an accessory that can be used to partially read the literal
     * or write if the graph is WriteGraph. The returned accessory is closed
     * automatically when the transaction ends. The modifications are written
     * back to graph at transaction completion.
     * 
     * @param <T> is typed accessory for modifying the literal value. 
     * @param resource is the literal containing the value.
     * @return interface to partially read and write the literal value.
     * @throws DatabaseException
     */
    <T extends Accessor> T getAccessor(Resource resource) throws DatabaseException;
    
    /**
     * Returns an RandomAccessBinary that can be used to partially read the literal
     * or write if the graph is WriteGraph. The returned interface is closed
     * automatically when the transaction ends. The modifications are written
     * back to graph at transaction completion.
     * 
     * @param <T>
     * @param resource is the literal containing the value.
     * @return interface to partially read and write the literal value.
     * @throws DatabaseException
     */
    RandomAccessBinary getRandomAccessBinary(Resource resource) throws DatabaseException;
//    @SuppressWarnings("rawtypes")
//	Function getValueFunction(Resource resource) throws DatabaseException;

    /**
     * Get the value associated with a graph {@link Resource}, using a possible context object and
     * a default data type binding. An exception is thrown, if the value is not found.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param r  A graph resource with which the value is associated
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @return  The value of the graph node.
     * @throws DoesNotContainValueException  No value is associated with the graph node.
     * @throws DatabaseException  Other errors, such as an error in casting the value to the return type or
     *         a runtime error in the value function.
     * @see #getValue2(Resource, Object, Binding)
     * @see #getPossibleValue2(Resource, Object)
     */
    <T> T getValue2(Resource subject, Object context) throws DatabaseException;
    
    /**
     * Get the value associated with a graph {@link Resource}, using a possible context object and
     * a default data type binding as a Variant. An exception is thrown, if the value is not found.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param r  A graph resource with which the value is associated
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @return  The value of the graph node.
     * @throws DoesNotContainValueException  No value is associated with the graph node.
     * @throws DatabaseException  Other errors, such as an error in casting the value to the return type or
     *         a runtime error in the value function.
     * @see #getValue2(Resource, Object, Binding)
     * @see #getPossibleValue2(Resource, Object)
     */
    Variant getVariantValue2( Resource r, Object context ) throws DatabaseException;
    
    /**
     * Get a possible value associated with a graph {@link Resource}, using a possible context object and
     * a desired value binding. Unlike {@link #getValue2(Resource, Object)}, this methods
     * returns {@code null} for missing values instead of throwing an exception.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param r  A graph resource with which the value is associated
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @return  The value of the graph node.
     * @throws DatabaseException  Usually should not happen. A null value is returned for possible errors.
     * @see #getValue2(Resource, Object)
     * @see #getPossibleValue2(Resource, Object, Binding)
     */
    <T> T getPossibleValue2(Resource subject, Object context) throws DatabaseException;
    
    /**
     * Get the value associated with a graph {@link Resource}, using a possible context object and
     * a desired value binding. An exception is thrown, if the value is not found.
     * <p>
     * The following methods are tried in order to retrieve the value:
     * <ol>
     *   <li>If the given resource is a {@code L0.Literal}, the value of the literal is returned, using the binding specified by {@code binding}.</li>
     *   <li>If the resource is a {@code L0.ExternalValue}, the value is acquired using
     *       {@link ReflectionUtils#getValue(String)} with the resource URI.</li>
     *   <li>If the resource is associated with a suitable value converter with relation {@code L0.ConvertsToValueWith}
     *       (see {@link #requestValueFunction(Resource)}), the value function is called with the graph, the resource
     *       and the context object.</li>
     * </ul>
     * 
     * @param r  A graph resource with which the value is associated
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @param binding  A binding for the value type
     * @return  The value of the graph node.
     * @throws DoesNotContainValueException  No value is associated with the graph node.
     * @throws DatabaseException  Other errors, such as an error in casting the value to the return type or
     *         a runtime error in the value function.
     * @see #getValue2(Resource, Object)
     * @see #getPossibleValue2(Resource, Object, Binding)
     */
    <T> T getValue2(Resource subject, Object context, Binding binding) throws DatabaseException;
    
    /**
     * Get a possible value associated with a graph {@link Resource}, using a possible context object and
     * a desired value binding. Unlike {@link #getValue2(Resource, Object, Binding)}, this methods
     * returns {@code null} for missing values instead of throwing an exception.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param r  A graph resource with which the value is associated
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @param binding  A binding for the value type
     * @return  The value of the graph node.
     * @throws DatabaseException  Usually should not happen. A null value is returned for possible errors.
     * @see #getValue2(Resource, Object, Bindinge)
     * @see #getPossibleValue2(Resource, Object)
     */
    <T> T getPossibleValue2(Resource subject, Object context, Binding binding) throws DatabaseException;
    
    /**
     * Get the single value associated with a graph resource related with a given subject.
     * An exception is thrown, if a unique object resource or a value for it is not found.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws NoSingleResultException  The number of suitable object resources is not equal to 1 (zero or greater than one) 
     * @throws DoesNotContainValueException  No value is associated with the graph node.
     * @throws DatabaseException  Other errors, such as an error in casting the value to the return type or
     *         a runtime error in the value function.
     * @see #getRelatedValue2(Resource, Resource, Binding)
     * @see #getRelatedValue2(Resource, Resource, Object)
     * @see #getRelatedValue2(Resource, Resource, Object, Binding)
     * @see #getPossibleRelatedValue2(Resource, Resource)
     */
    <T> T getRelatedValue2(Resource subject, Resource relation) throws DatabaseException;

    /**
     * Get a possible single value associated with a graph resource related with a given subject.
     * A {@code null} value is returned for missing values or other errors.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws DatabaseException  Usually should not happen. A {@code null} value is returned for possible errors.
     * @see #getPossibleRelatedValue2(Resource, Resource, Binding)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object, Binding)
     * @see #getRelatedValue2(Resource, Resource)
     */
    <T> T getPossibleRelatedValue2(Resource subject, Resource relation) throws DatabaseException;
    
    /**
     * Get a possible single value associated with a graph resource related with a given subject.
     * A {@code null} value is returned for missing values or other errors.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws DatabaseException  Usually should not happen. A {@code null} value is returned for possible errors.
     * @see #getPossibleRelatedValue2(Resource, Resource, Binding)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object, Binding)
     * @see #getRelatedValue2(Resource, Resource)
     */
    Variant getRelatedVariantValue2( Resource subject, Resource relation ) throws DatabaseException;
    
    /**
     * Get the single value associated with a graph resource related with a given subject, a possible context object.
     * An exception is thrown, if a unique object resource or a value for it is not found.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws NoSingleResultException  The number of suitable object resources is not equal to 1 (zero or greater than one) 
     * @throws DoesNotContainValueException  No value is associated with the graph node.
     * @throws DatabaseException  Other errors, such as an error in casting the value to the return type or
     *         a runtime error in the value function.
     * @see #getRelatedValue2(Resource, Resource)
     * @see #getRelatedValue2(Resource, Resource, Binding)
     * @see #getRelatedValue2(Resource, Resource, Object, Binding)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object)
     */
    <T> T getRelatedValue2(Resource subject, Resource relation, Object context) throws DatabaseException;
    
    /**
     * Get the single value associated with a graph resource related with a given subject, a possible context object.
     * An exception is thrown, if a unique object resource or a value for it is not found.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws NoSingleResultException  The number of suitable object resources is not equal to 1 (zero or greater than one) 
     * @throws DoesNotContainValueException  No value is associated with the graph node.
     * @throws DatabaseException  Other errors, such as an error in casting the value to the return type or
     *         a runtime error in the value function.
     * @see #getRelatedValue2(Resource, Resource, Object)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object)
     */
    Variant getRelatedVariantValue2( Resource subject, Resource relation, Object context ) throws DatabaseException;
    
    /**
     * Get a possible single value associated with a graph resource related with a given subject and a possible context object.
     * A {@code null} value is returned for missing values or other errors.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws DatabaseException  Usually should not happen. A {@code null} value is returned for possible errors.
     * @see #getPossibleRelatedValue2(Resource, Resource)
     * @see #getPossibleRelatedValue2(Resource, Resource, Binding)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object, Binding)
     * @see #getRelatedValue2(Resource, Resource, Object)
     */
    <T> T getPossibleRelatedValue2(Resource subject, Resource relation, Object context) throws DatabaseException;

    /**
     * Get the single value associated with a graph resource related with a given subject and
     * a desired value binding. An exception is thrown, if a unique object resource or a value for it is not found.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @param binding  A binding for the value type
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws NoSingleResultException  The number of suitable object resources is not equal to 1 (zero or greater than one) 
     * @throws DoesNotContainValueException  No value is associated with the graph node.
     * @throws DatabaseException  Other errors, such as an error in casting the value to the return type or
     *         a runtime error in the value function.
     * @see #getRelatedValue2(Resource, Resource)
     * @see #getRelatedValue2(Resource, Resource, Object)
     * @see #getRelatedValue2(Resource, Resource, Object, Binding)
     * @see #getPossibleRelatedValue2(Resource, Resource, Binding)
     */
    <T> T getRelatedValue2(Resource subject, Resource relation, Binding binding) throws DatabaseException;

    /**
     * Get a possible single value associated with a graph resource related with a given subject and
     * a desired value binding. A {@code null} value is returned for missing values or other errors.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @param binding  A binding for the value type
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws DatabaseException  Usually should not happen. A {@code null} value is returned for possible errors.
     * @see #getPossibleRelatedValue2(Resource, Resource)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object, Binding)
     * @see #getRelatedValue2(Resource, Resource, Binding)
     */
    <T> T getPossibleRelatedValue2(Resource subject, Resource relation, Binding binding) throws DatabaseException;
    
    /**
     * Get the single value associated with a graph resource related with a given subject, a possible context object and
     * a desired value binding. An exception is thrown, if a unique object resource or a value for it is not found.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @param binding  A binding for the value type
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws NoSingleResultException  The number of suitable object resources is not equal to 1 (zero or greater than one) 
     * @throws DoesNotContainValueException  No value is associated with the graph node.
     * @throws DatabaseException  Other errors, such as an error in casting the value to the return type or
     *         a runtime error in the value function.
     * @see #getRelatedValue2(Resource, Resource)
     * @see #getRelatedValue2(Resource, Resource, Binding)
     * @see #getRelatedValue2(Resource, Resource, Object)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object, Binding)
     */
    <T> T getRelatedValue2(Resource subject, Resource relation, Object context, Binding binding) throws DatabaseException;

    /**
     * Get a possible single value associated with a graph resource related with a given subject, a possible context object and
     * a desired value binding. A {@code null} value is returned for missing values or other errors.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     * 
     * @param subject  A graph resource that indicates the subject node
     * @param relation  A graph resource that indicates the predicate of a relation
     * @param context  A context object that is used for acquiring the value, can be {@code null}
     * @param binding  A binding for the value type
     * @return  The value of the unique graph node that is asserted as the object of the given subject and predicate.
     * @throws DatabaseException  Usually should not happen. A {@code null} value is returned for possible errors.
     * @see #getPossibleRelatedValue2(Resource, Resource)
     * @see #getPossibleRelatedValue2(Resource, Resource, Binding)
     * @see #getPossibleRelatedValue2(Resource, Resource, Object)
     * @see #getRelatedValue2(Resource, Resource, Object, Binding)
     */
    <T> T getPossibleRelatedValue2(Resource subject, Resource relation, Object context, Binding binding) throws DatabaseException;
    
    /**
     * Get the parsed SCL type declaration from the string value of a graph resource that is the unique object of a
     * given subject and predicate.
     * <p>
     * See {@link #getValue2(Resource, Object, Binding)} for more details.
     *  
     * @param subject  A resource that indicates the subject of a statement
     * @param relation  A resource that indicates the predicate of the statement
     * @return  A compiled SCL type definition of the string value of a unique object resource
     * @throws DatabaseException  For any errors in locating a unique value or compiling it as an SCL type statement
     */
    Type getRelatedValueType(Resource subject, Resource relation) throws DatabaseException;
    
    /*
     * @deprecated use <code>setHintValue</code> instead
     */
    @Deprecated
    boolean setSynchronous(boolean value);
    
    /*
     * @deprecated use <code>getHintValue</code> instead
     */
    @Deprecated
    boolean getSynchronous();
    
    <T> T getHintValue(String key);
    <T> GraphHints setHintValue(String key, T value);
    GraphHints setHints(GraphHints hints);

    boolean isImmutable(Resource resource) throws DatabaseException;

    int thread();

}
