package org.simantics.db.layer0.scl;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.IndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.scl.CompileComputationalValueRequest.CompilationContext;
import org.simantics.db.layer0.util.RuntimeEnvironmentRequest;
import org.simantics.layer0.Layer0;
import org.simantics.scl.compiler.elaboration.expressions.EVariable;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.environment.Environments;
import org.simantics.scl.compiler.runtime.RuntimeEnvironment;
import org.simantics.scl.compiler.top.SCLExpressionCompilationException;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.Function1;

/**
 * Compiles an SCL expression that is attached to a resource
 * 
 * @author Antti Villberg
 * @since 1.51.0
 */
public class CompileComputationalValueRequest extends AbstractExpressionCompilationRequest<CompilationContext, Object> {

    public static class CompilationContext extends AbstractExpressionCompilationContext {
        public CompilationContext(RuntimeEnvironment runtimeEnvironment) {
            super(runtimeEnvironment);
        }
    }

    protected final Resource resource;

    public CompileComputationalValueRequest(Resource resource) {
        this.resource = resource;
    }

    public static <T> T compileAndEvaluate(ReadGraph graph, Resource resource) throws DatabaseException {
        SCLContext sclContext = SCLContext.getCurrent();
        Object oldGraph = sclContext.get("graph");
        try {
        	Function1<Object,Object> exp = (Function1<Object,Object>)graph.syncRequest(new CompileComputationalValueRequest(resource), TransientCacheListener.instance());
            sclContext.put("graph", graph);
            return (T)exp.apply(resource);
        } catch (DatabaseException e) {
            throw (DatabaseException)e;
        } catch (Throwable t) {
            throw new DatabaseException(t);
        } finally {
            sclContext.put("graph", oldGraph);
        }
    }

    @Override
    protected String getExpressionText(ReadGraph graph)
            throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        return graph.getRelatedValue(resource, L0.SCLExternalValue_expression, Bindings.STRING);
    }

    protected Resource getIndexRoot(ReadGraph graph) throws DatabaseException {
        return graph.syncRequest(new IndexRoot(resource));
    }

    @Override
    protected Type getExpectedType(ReadGraph graph, CompilationContext context)
            throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        String valueType = graph.getPossibleRelatedValue(resource, L0.HasValueType, Bindings.STRING);
        if(valueType != null) {
            try {
                return Environments.getType(context.runtimeEnvironment.getEnvironment(), valueType);
            } catch (SCLExpressionCompilationException e) {
                e.printStackTrace();
            }
        }
        return super.getExpectedType(graph, context);
    }

    @Override
    protected CompilationContext getCompilationContext(ReadGraph graph)
            throws DatabaseException {
        Resource indexRoot = getIndexRoot(graph);
        RuntimeEnvironment runtimeEnvironment = graph.syncRequest(new RuntimeEnvironmentRequest(indexRoot));
        return new CompilationContext(runtimeEnvironment);
    }

    @Override
    protected Type getContextVariableType() {
        return RESOURCE; 
    }

    @Override
    protected Expression getVariableAccessExpression(
            ReadGraph graph,
            CompilationContext context,
            org.simantics.scl.compiler.elaboration.expressions.Variable contextVariable,
            String name) throws DatabaseException {
        if(name.equals("self"))
            return new EVariable(contextVariable);
        else
            return null;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((resource == null) ? 0 : resource.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        CompileComputationalValueRequest other = (CompileComputationalValueRequest) obj;
        if (resource == null) {
            if (other.resource != null)
                return false;
        } else if (!resource.equals(other.resource))
            return false;
        return true;
    }


}
