/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import org.simantics.databoard.Bindings;
import org.simantics.db.DevelopmentKeys;
import org.simantics.db.RelationInfo;
import org.simantics.db.Resource;
import org.simantics.db.common.exception.DebugException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.IntProcedureAdapter;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.request.RequestFlags;
import org.simantics.utils.Development;

/*
 * Size analysis:
 * java 8 byte
 * id 8 byte
 * statusOrException 4 byte
 * p1 = 4 byte
 * p2OrParents = 4 byte
 * result = 4 byte 
 * 
 * total 32byte
 * 
 */

public final class Objects extends CollectionBinaryQuery<IntProcedure> implements IntProcedure {

    public Objects(final int r1, final int r2) {
        super(r1, r2);
    }

    @Override
    final public void removeEntry(QueryProcessor provider) {
        provider.cache.remove(this);
    }

    final static private IntArray getAssertionMap(ReadGraphImpl graph, final int r1, final int r2, final Objects entry) throws DatabaseException {

        class AssertionMapProc implements IntProcedure {

            boolean first = true;

            private IntArray result;

            public void addStatement(int s, int p, int o) {

                if(result.size() == 0) {
                    result.add(s);
                    result.add(p);
                    result.add(o);
                } else {
                    for(int i = 0;i < result.sizeOrData ; i+=3) {
                        int existingP = result.data[i+1];
                        if(p == existingP) {
                            int existingO = result.data[i+2];
                            if(existingO == o) return;
                        }
                    }
                    result.add(s);
                    result.add(p);
                    result.add(o);
                }

            }

            @Override
            public void execute(ReadGraphImpl graph, int type) throws DatabaseException {
                if(result == null) {
                    result = QueryCacheBase.resultAssertedStatements(graph, type, r2, entry, null);
                } else {
                    if (first) {
                        IntArray ia = result;
                        result = new IntArray();
                        if(ia.data != null) {
                            for(int i = 0;i < ia.sizeOrData ; i+=3) addStatement(ia.data[i],ia.data[i+1],ia.data[i+2]);
                        }
                        first = false;
                    }
                    IntArray ia = QueryCacheBase.resultAssertedStatements(graph, type, r2, entry, null);
                    if(ia.data != null) {
                        for(int i = 0;i < ia.sizeOrData ; i+=3) addStatement(ia.data[i],ia.data[i+1],ia.data[i+2]);
                    }
                }
            }

            @Override
            public void finished(ReadGraphImpl graph) {
            }

            @Override
            public void exception(ReadGraphImpl graph, Throwable throwable) {
            }

        }

        AssertionMapProc amp = new AssertionMapProc();

        // This dependency could be cut
        QueryCache.runnerPrincipalTypes(graph, r1, entry, null, amp);

        return amp.result;

    }

    final static private void forSingleAssertion(ReadGraphImpl graph, final int r1, final int r2, final Objects parent, final IntProcedure procedure) throws DatabaseException {

        IntArray map = getAssertionMap(graph, r1, r2, parent);
        if(map == null) {
            procedure.finished(graph);
            return;
        }

        int size = map.size();
        if(size == 3) {
            int value = map.data[2];
            procedure.execute(graph, value);
            procedure.finished(graph);
        } else if(size == 0) {
            procedure.finished(graph);
        } else {

            int candidateS = map.data[0];
            int candidateO = map.data[2];

            IntSet candidateIs = null;
            try {
                candidateIs = QueryCache.resultSuperTypes(graph, candidateS, parent, null);
            } catch (DatabaseException e) {
                procedure.exception(graph, e);
                return;
            }

            for(int i=3;i<map.size();i+=3) {

                int nextS = map.data[i];
                int nextO = map.data[i+2];

                if(nextS != candidateS) {

                    if(candidateIs.contains(nextS)) {

                        // Next is a super type of candidate => ignore next

                    } else {

                        IntSet nextIs = null;
                        try {
                            nextIs = QueryCache.resultSuperTypes(graph, nextS, parent, null);
                        } catch (DatabaseException e) {
                            procedure.exception(graph, e);
                            return;
                        }

                        if(nextIs.contains(candidateS)) {

                            // Candidate is a super type of next => next is the new candidate

                            candidateS = nextS;
                            candidateO = nextO;
                            candidateIs = nextIs;

                        } else {

                            // candidate and next are unrelated => error
                            ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has conflicting assertions " + r1 + ", " + r2 + " " + map , r1);
                            procedure.exception(graph, exception);
                            return;

                        }

                    }

                }

            }

            procedure.execute(graph, candidateO);
            procedure.finished(graph);

        }

    }

    final static InternalProcedure<IntSet> NOP = new InternalProcedure<IntSet>() {

        @Override
        public void execute(ReadGraphImpl graph, IntSet result) {
        }

        @Override
        public void exception(ReadGraphImpl graph, Throwable throwable) {
        }

    };

    final static TripleIntProcedure NOPT = new TripleIntProcedure() {


        @Override
        public void exception(ReadGraphImpl graph, Throwable throwable) {
        }

        @Override
        public void execute(ReadGraphImpl graph, int s, int p, int o) {
        }

        @Override
        public void finished(ReadGraphImpl graph) {
        }

    };

    // Search for one statement
    final public void computeFunctionalIndex(ReadGraphImpl graph, final QueryProcessor provider, final RelationInfo ri, final IntProcedure procedure) throws DatabaseException {
        computeFunctionalIndex(graph, r1(), r2(), this, ri, procedure);
    }

    // Search for one statement
    final static public void computeFunctionalIndex(ReadGraphImpl graph, final int r1, final int r2, final Objects parent, final RelationInfo ri, final IntProcedure procedure) throws DatabaseException {

        if(ri.isFinal) {

            int result = graph.processor.querySupport.getFunctionalObject(r1, r2);

            if(result == 0) {

                // Check for assertions
                forSingleAssertion(graph, r1, r2, parent, procedure);

            } else if (result == -1) {

                graph.processor.querySupport.getObjects(graph, r1, r2, new IntProcedure() {

                    @Override
                    public void execute(ReadGraphImpl graph, int i) throws DatabaseException {
                        procedure.execute(graph, i);
                    }

                    @Override
                    public void exception(ReadGraphImpl graph, Throwable t) {
                        if(DebugException.DEBUG) new DebugException(t).printStackTrace();
                    }

                    @Override
                    public void finished(ReadGraphImpl graph) {
                    }

                });

                // Check for assertions
                forSingleAssertion(graph, r1, r2, parent, procedure);

            } else {

                // If functional relation was found there is no need to check assertions
                procedure.execute(graph, result);
                procedure.finished(graph);

            }

        } else {

            // Note! The dependency is intentionally cut!
            IntSet direct = QueryCache.resultDirectPredicates(graph, r1, null, null);
            direct.forEach(graph, new SyncIntProcedure() {

                /*
                 * 0 = not found
                 * 1 = found
                 * 2 = exception
                 */
                int found = 0;

                @Override
                public void run(ReadGraphImpl graph) throws DatabaseException {

                    if(found == 1) {

                        procedure.finished(graph);

                    } else if(found == 0) {

                        // Check for assertions
                        forSingleAssertion(graph, r1, r2, parent, procedure);

                    }

                }

                @Override
                public void execute(ReadGraphImpl graph, final int pred) throws DatabaseException {

                    if(found > 0)
                        return;

                    if(pred == r2) {

                        // Note! The dependency is intentionally cut!
                        QueryCache.runnerDirectObjects(graph, r1, pred, null, null, new IntProcedure() {

                            @Override
                            public void execute(ReadGraphImpl graph, int i) throws DatabaseException {

                                if(found == 0) {

                                    procedure.execute(graph, i);
                                    found = 1;

                                } else {

                                    ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has more than one statement (r1=" + r1 + ", r2=" + r2 + ").", r1);
                                    procedure.exception(graph, exception);
                                    found = 2;

                                }

                            }

                            @Override
                            public void finished(ReadGraphImpl graph) {
                            }

                            @Override
                            public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {

                                procedure.exception(graph, t);
                                found = 2;

                            }

                        });

                    } else {

                        QueryCache.runnerSuperRelations(graph, pred, parent, null, new InternalProcedure<IntSet>() {

                            @Override
                            public void execute(ReadGraphImpl graph, IntSet result) throws DatabaseException {

                                if(found > 0)
                                    return;

                                if(result.contains(r2)) {

                                    // Note! The dependency is intentionally cut!
                                    QueryCache.runnerDirectObjects(graph, r1, pred, null, null, new IntProcedure() {

                                        @Override
                                        public void execute(ReadGraphImpl graph, int i) throws DatabaseException {

                                            if(found == 0) {

                                                procedure.execute(graph, i);
                                                found = 1;

                                            } else {

                                                ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has more than one statement (r1=" + r1 + ", r2=" + r2 + ").", r1);
                                                procedure.exception(graph, exception);
                                                found = 2;

                                            }

                                        }

                                        @Override
                                        public void finished(ReadGraphImpl graph) {
                                        }

                                        @Override
                                        public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                                            procedure.exception(graph, t);
                                            found = 2;
                                        }

                                    });

                                }

                            }

                            @Override
                            public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                                procedure.exception(graph, t);
                                found = 2;
                            }

                        });

                    }

                }

                @Override
                public void finished(ReadGraphImpl graph) throws DatabaseException {
                    dec(graph);
                }

            });


        }

    }

    final static private void forAssertions(ReadGraphImpl graph, final int r1, final int r2, final Objects parent, final IntProcedure procedure) throws DatabaseException {

        // Note! The dependency is intentionally cut!
        QueryCache.runnerPrincipalTypes(graph, r1, null, null, new SyncIntProcedure() {

            @Override
            public void run(ReadGraphImpl graph) throws DatabaseException {
                procedure.finished(graph);
            }

            TripleIntProcedure proc = new TripleIntProcedure() {

                @Override
                public void execute(ReadGraphImpl graph, int s, int p, int o) throws DatabaseException {
                    procedure.execute(graph, o);
                }

                @Override
                public void finished(ReadGraphImpl graph) throws DatabaseException {
                    dec(graph);
                }

                @Override
                public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                    if(DebugException.DEBUG) new DebugException(t).printStackTrace();
                    procedure.exception(graph, t);
                    dec(graph);
                }

            };

            @Override
            public void execute(ReadGraphImpl graph, int type) throws DatabaseException {

                inc();
                QueryCache.runnerAssertedStatements(graph, type, r2, parent, null, proc);

            }

            @Override
            public void finished(ReadGraphImpl graph) throws DatabaseException {
                dec(graph);
            }

        });


    }

    final public static void computeNotFunctionalFinalIndex(ReadGraphImpl graph, final int r1, final int r2, final QueryProcessor provider, RelationInfo ri, AsyncMultiProcedure<Resource> procedure) {
        throw new Error();
    }

    final public void computeNotFunctionalIndex(ReadGraphImpl graph, RelationInfo ri, final IntProcedure procedure) throws DatabaseException {
        computeNotFunctionalIndex(graph, r1(), r2(), this, ri, procedure);
    }

    final static public void computeNotFunctionalIndex(ReadGraphImpl graph, final int r1, final int r2, final Objects parent, RelationInfo ri, final IntProcedure procedure) throws DatabaseException {

        if(ri.isFinal) {

            graph.processor.querySupport.getObjects(graph, r1, r2, new IntProcedure() {

                @Override
                public void execute(ReadGraphImpl graph, int i) throws DatabaseException {
                    procedure.execute(graph, i);
                }

                @Override
                public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                    if(DebugException.DEBUG) new DebugException(t).printStackTrace();
                    procedure.exception(graph, t);
                }

                @Override
                public void finished(ReadGraphImpl graph) {
                }

            });

            if(ri.isAsserted) {
                forAssertions(graph, r1, r2, parent, procedure);
            } else {
                procedure.finished(graph);
            }

        } else {

            // Note! The dependency is intentionally cut!
            IntSet direct = QueryCache.resultDirectPredicates(graph, r1, null, null);
            direct.forEach(graph, new SyncIntProcedure() {

                @Override
                public void run(ReadGraphImpl graph) throws DatabaseException {
                    forAssertions(graph, r1, r2, parent, procedure);
                }

                @Override
                public void execute(ReadGraphImpl graph, final int pred) throws DatabaseException {

                    if(pred == r2) {

                        // Note! The dependency is intentionally cut!
                        QueryCache.runnerDirectObjects(graph, r1, pred, null, null, new IntProcedure() {

                            @Override
                            public void execute(ReadGraphImpl graph, int i) throws DatabaseException {
                                procedure.execute(graph, i);
                            }

                            @Override
                            public void finished(ReadGraphImpl graph) throws DatabaseException {
                            }

                            @Override
                            public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                                procedure.exception(graph, t);
                            }

                        });

                    } else {

                        try {

                            IntSet result = QueryCache.resultSuperRelations(graph, pred, parent, null);
                            if(result.contains(r2)) {

                                inc();

                                // Note! The dependency is intentionally cut!
                                QueryCache.runnerDirectObjects(graph, r1, pred, null, null, new IntProcedure() {

                                    @Override
                                    public void execute(ReadGraphImpl graph, int i) throws DatabaseException {
                                        procedure.execute(graph, i);
                                    }

                                    @Override
                                    public void finished(ReadGraphImpl graph) throws DatabaseException {
                                        dec(graph);
                                    }

                                    @Override
                                    public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                                        if(DebugException.DEBUG) new DebugException(t).printStackTrace();
                                        procedure.exception(graph, t);
                                        dec(graph);
                                    }

                                });

                            }

                        } catch (Throwable e) {
                            procedure.exception(graph, e);
                        }

                    }

                }

                @Override
                public void finished(ReadGraphImpl graph) throws DatabaseException {
                    dec(graph);
                }

            });

        }

    }

    public Object compute(ReadGraphImpl graph, final IntProcedure procedure) throws DatabaseException {
        computeForEach(graph, r1(), r2(), this, procedure);
        return getResult();
    }

    public static void computeForEach(ReadGraphImpl graph, final int r1, final int r2, final Objects entry, final IntProcedure procedure_) throws DatabaseException {

        if(Development.DEVELOPMENT) {
            if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_PERFORM_QUERY, Bindings.BOOLEAN)) {
                Development.log("PQ Objects " + r1 + " - " + r2);
            }
        }

        IntProcedure procedure = entry != null ? entry : procedure_;

        RelationInfo ri = QueryCache.resultRelationInfoQuery(graph, r2, entry, null);
        graph.ensureLoaded(r1, r2);
        if(ri.isFunctional) {
            computeFunctionalIndex(graph, r1, r2, entry, ri, procedure);
        } else {
            computeNotFunctionalIndex(graph, r1, r2, entry, ri, procedure);
        }

        if(entry != null) entry.performFromCache(graph, procedure_);

    }

    @Override
    public String toString() {
        return "Objects[" + r1() + " - " + r2() + "]";
    }

    @Override
    public Object performFromCache(ReadGraphImpl graph, final IntProcedure procedure) throws DatabaseException {

        assert(isReady());

        if(handleException(graph, procedure)) return getResult();

        final IntArray value = (IntArray)getResult();
        if(value.data == null) {
            if(value.sizeOrData != IntArray.NO_DATA) procedure.execute(graph, value.sizeOrData);
        } else {
            for(int i = 0;i < value.sizeOrData ; i++) procedure.execute(graph, value.data[i]);
        }

        procedure.finished(graph);

        return value;

    }

    @Override
    public void recompute(ReadGraphImpl graph) throws DatabaseException {

        compute(graph, new IntProcedureAdapter() {

            @Override
            public void finished(ReadGraphImpl graph) {
            }

            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
                new Error("Error in recompute.", t).printStackTrace();
            }

        });

    }

    @Override
    public int type() {
        return RequestFlags.IMMEDIATE_UPDATE;
    }

    @Override
    boolean isImmutable(ReadGraphImpl graph) {
        return graph.processor.isImmutable(r1());
    }

    @Override
    public void execute(ReadGraphImpl graph, int i) throws DatabaseException {
        IntArray value = (IntArray)getResult();
        synchronized(value) {
            value.add(i);
        }
    }

    @Override
    public void finished(ReadGraphImpl graph) throws DatabaseException {
        setReady();
    }

    @Override
    public void exception(ReadGraphImpl graph, Throwable throwable) throws DatabaseException {
        except(throwable);
    }

    @Override
    public void serializeValue(QuerySerializer serializer) {
        IntArray is = getResult();
        is.serialize(serializer);
    }
    
}
