/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import java.util.Arrays;
import java.util.HashMap;

import junit.framework.TestCase;

import org.junit.Test;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.annotations.Union;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.impl.HashMapBinding;
import org.simantics.databoard.binding.impl.OptionalBindingDefault;
import org.simantics.databoard.parser.repository.DataValueRepository;

public class ValueParserSuccessTests extends TestCase { 
	
	public Object parse(Binding binding, String text) throws Exception {
		return new DataValueRepository().translate(text, binding);
	}
	
	@Test
	public void testPrimitives() throws Exception {
		assertEquals(Boolean.TRUE, parse(Bindings.BOOLEAN, "true"));
		assertEquals(Boolean.FALSE, parse(Bindings.BOOLEAN, "false"));
		
		assertEquals(123, parse(Bindings.INTEGER, "123"));
		assertEquals(-234234, parse(Bindings.INTEGER, "-234234"));
		
		assertEquals(12345678912345L, parse(Bindings.LONG, "12345678912345"));
		
		assertEquals(1.2345, parse(Bindings.DOUBLE, "1.2345"));		
		assertEquals(-3.0, parse(Bindings.DOUBLE, "-3"));
		assertEquals(1e-10, parse(Bindings.DOUBLE, "1e-10"));
		
		assertEquals(2.3f, parse(Bindings.FLOAT, "2.3"));
		
		assertEquals("Hello World!", parse(Bindings.STRING, "\"Hello World!\""));
		assertEquals("\n\t\r\\\"", parse(Bindings.STRING, "\"\\n\\t\\r\\\\\\\"\""));
		assertEquals("This\ntext\nspans\nmultiple\nlines.", 
				parse(Bindings.STRING, "\"\"\"This\ntext\nspans\nmultiple\nlines.\"\"\""));
		assertEquals("\u0008", parse(Bindings.STRING, "\"\b\""));
	}
	
	public void assertArraysEqual(int[] expected, int[] actual) {
		if(!Arrays.equals(expected, actual))		
			failNotEquals(null, expected, actual);
	}
	
	public void testArray() throws Exception {
		assertArraysEqual(new int[] {1,2,3}, (int[])parse(Bindings.INT_ARRAY, "[1,2,3]"));
	}
	
	public void testOptional() throws Exception {
		assertEquals(null, 
				(Integer)parse(new OptionalBindingDefault(Bindings.INTEGER), "null"));
		assertEquals((Integer)1, 
				(Integer)parse(new OptionalBindingDefault(Bindings.INTEGER), "1"));
	}
	
	static public class MyRecord {
		public int x;
		public String y;
		
		public MyRecord(int x, String y) {
			this.x = x;
			this.y = y;
		}			
	}
	
	public void testRecord() throws Exception {
		MyRecord record = (MyRecord)parse(Bindings.getBinding(MyRecord.class), "{ y = \"asd\", x = 5 }");
		assertEquals(5, record.x);
		assertEquals("asd", record.y);
	}
	
	public static class MyRecord2 {
		public int x;
		public String y;
	}
	
	public void testRecord2() throws Exception {
		MyRecord2 record = (MyRecord2)parse(Bindings.getBinding(MyRecord2.class), "{ y = \"asd\", x = 5 }");
		assertEquals(5, record.x);
		assertEquals("asd", record.y);
	}
	
	public void testTuple() throws Exception {
		MyRecord record = (MyRecord)parse(Bindings.getBinding(MyRecord.class), "( 5, \"asd\")");
		assertEquals(5, record.x);
		assertEquals("asd", record.y);
	}
	
	@Union({A.class, C.class})
	interface Union1 {		
	}
	
	static public class A implements Union1 {}
	static public class C implements Union1 { 
		public float x; 
		public int y;
		public C(float x, int y) {
			this.x = x;
			this.y = y;
		}		
	}
	
	public void testUnion() throws Exception {
		Union1 union = (Union1)parse(Bindings.getBinding(Union1.class), "C (1.2, 3)");
		assertTrue(union instanceof C);
		assertEquals(1.2f, ((C)union).x);
		assertEquals(3, ((C)union).y);
	}
	
	@SuppressWarnings("rawtypes")
    public void testMap() throws Exception {
		HashMap map = (HashMap)parse(
				new HashMapBinding(
						Bindings.getBinding(String.class),
						Bindings.getBinding(Integer.class)
						),				 
				"map { \"1\" = 1, \"2\" = 2 }");
		assertEquals(2, map.size());
		assertEquals(1, map.get("1"));
		assertEquals(2, map.get("2"));
	}
	
}
