/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import java.io.File;
import java.util.TreeMap;

import org.simantics.databoard.Accessors;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.accessor.file.FileMapAccessor;
import org.simantics.databoard.accessor.file.FileRecordAccessor;
import org.simantics.databoard.accessor.file.FileVariantAccessor;
import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.util.RandomValue;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.LongType;

public class PerformanceTest {	
	
	public static class FloatSample {
		public float value;
	}
	
	public static class IntegerSample {
		public int value;
	}
	
	public static class FloatRecord {
		public Object id;
		
		public TreeMap<String, String> displayNames;
		
		public SamplingParameters samplingParameters;
		
		public TreeMap<Long, FloatSample> samples;
	}

	public static class IntegerRecord {
		public Object id;
		
		public TreeMap<String, String> displayNames;
		
		public SamplingParameters samplingParameters;
		
		public TreeMap<Long, IntegerSample> samples;
	}
	
	public static class SamplingParameters {
		public @Optional Object deadband;
		public @Optional Object interval;
	}
	
	public static class LocalizedText {
		public String locale;
		public String text;
	}
	
	public static void main(String[] args) throws Exception {
		// Init
		Datatype time_type 					= new LongType("ms");
		Binding time_binding				= Bindings.getBinding( Long.class );
		Binding float_sample_binding 		= Bindings.getBinding( FloatSample.class);
		Binding integer_sample_binding 		= Bindings.getBinding( IntegerSample.class);
		Binding float_record_binding 		= Bindings.getBinding( FloatRecord.class );
		Binding integer_record_binding 		= Bindings.getBinding( IntegerRecord.class );
		
		// Test variables
		int var_count = 1000;
		long interval = 10;
		long duration = 10000000;//10*60*60*1000; // ms
		int  block_size = 100; 
		long steps = duration / interval;
		long iterations = steps / block_size;
		System.out.println("Performance Test");
		System.out.println(" o variables = "+var_count);
		System.out.println(" o duration = "+duration/1000+" s");
		System.out.println(" o interval = "+interval+" ms");
		System.out.println(" o samples_in_write = "+block_size);
		System.out.println(" o At each step, a single sample is appended.");
		
		
		// Prepare files
		System.out.println(" o Initialization.");
		FileRecordAccessor[] variables = new FileRecordAccessor[var_count];
		FileMapAccessor[] maps = new FileMapAccessor[var_count];
		for (int i=0; i<var_count; i++) {
			// Create a file
			// <variableId>.dat
			File file = File.createTempFile("variable"+i, ".tmp");
			file.deleteOnExit();
			FileVariantAccessor fa = Accessors.createFile(file);
						
			// Make it a record
			Binding record_binding = i<var_count/2 ? float_record_binding : integer_record_binding;			
			Object empty_record = record_binding.createDefault();
			fa.setContentValue(record_binding, empty_record);			
			
			// Get accessors to the record and its field "samples"
			FileRecordAccessor ra = fa.getContentAccessor();
			FileMapAccessor map = ra.getFieldAccessor("samples");			
			variables[i] = ra;
			maps[i] = map;
		}
		
//		System.out.println(" o variable1 = "+variables[0].type().toString());
		
		RandomValue rv = new RandomValue();
		FloatSample block1[] = new FloatSample[block_size];
		IntegerSample block2[] = new IntegerSample[block_size];
		for (int i=0; i<block_size; i++) {
			block1[i] = new FloatSample();
			block2[i] = new IntegerSample();
		}
		Long times[] = new Long[block_size];
		
		long startTime = System.currentTimeMillis();
		long reportTime = startTime;
		Long simulationTime = 0L;
		for (int i=0; i<iterations; i++) {
			
			for (int vi=0; vi<var_count; vi++) {
				FileMapAccessor map = maps[vi];
				Binding sample_binding = vi<var_count/2 ? float_sample_binding : integer_sample_binding;
				
				// Create [block_size] samples
				if (sample_binding == float_sample_binding) {
					// Create [block_size] random float samples
					for (int j=0; j<block_size; j++) {
						block1[j].value = rv.getRandom().nextFloat();
						times[j] = simulationTime + j * interval;
					}
					// Append [block_size] float samples
					map.putAllNoflush(time_binding, sample_binding, times, block1); 
				} else {
					// Create [block_size] random integer samples
					for (int j=0; j<block_size; j++) {
						block2[j].value = rv.getRandom().nextInt();
						times[j] = simulationTime + j * interval;
					}
					// Append [block_size] integer samples
					map.putAllNoflush(time_binding, sample_binding, times, block2); 
				}
				
			}

			// Buffers are flushed every 1000 steps
			long currentTime = System.currentTimeMillis();
			if (currentTime - reportTime > 1000L) {
				System.out.println("Report:");
				System.out.println(" o Simulation Time: "+simulationTime/1000+" s");
				System.out.println(" o Time elapsed: "+ (currentTime - startTime)/1000 + " s" );
				System.out.println(" o Samples in a variable: "+ maps[0].size());
//				for (int vi=0; vi<var_count; vi++) variables[vi].flush();
				
				reportTime = currentTime;
			}
			
			simulationTime += interval*block_size;
		}

		System.out.println("Simulation finished.");
		
		// Close files		
		System.out.println(" o Closing files.");
		for (int vi=0; vi<var_count; vi++) {
			variables[vi].flush();
			variables[vi].close();		
			variables[vi].file().delete();
		}		
		
		// Measure time		
		long endTime = System.currentTimeMillis();
		long elapsedTime = endTime - startTime;
		System.out.println(" o Elapsed Time: "+elapsedTime+" ms");
		
	}
	
}

