/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.factory;

import java.util.Map;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingConstructionException;
import org.simantics.databoard.binding.impl.BooleanArrayBinding;
import org.simantics.databoard.binding.impl.BooleanBindingDefault;
import org.simantics.databoard.binding.impl.ByteArrayBinding;
import org.simantics.databoard.binding.impl.ByteBindingDefault;
import org.simantics.databoard.binding.impl.DefaultMapBinding;
import org.simantics.databoard.binding.impl.DoubleArrayBinding;
import org.simantics.databoard.binding.impl.DoubleBindingDefault;
import org.simantics.databoard.binding.impl.FloatArrayBinding;
import org.simantics.databoard.binding.impl.FloatBindingDefault;
import org.simantics.databoard.binding.impl.IntArrayBinding;
import org.simantics.databoard.binding.impl.IntegerBindingDefault;
import org.simantics.databoard.binding.impl.LongArrayBinding;
import org.simantics.databoard.binding.impl.LongBindingDefault;
import org.simantics.databoard.binding.impl.ObjectArrayBinding;
import org.simantics.databoard.binding.impl.StringBindingDefault;
import org.simantics.databoard.binding.mutable.ContainerOptionalBinding;
import org.simantics.databoard.binding.mutable.UnionTaggedObjectBinding;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.type.MapType;
import org.simantics.databoard.type.OptionalType;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.type.UnionType;

/**
 * DefaultBindingScheme is a type to binding mapping that binds any DataType to an Object.
 * All resulting bindings typicaly immutable java classes.
 * 
 * DataType           | Class of the bound instance
 * ===================|==================
 * BooleanType        | Boolean.class
 * ByteType           | Byte.class
 * FloatType          | Float.class
 * DoubleType         | Double.class
 * IntegerType        | Int.class
 * LongType           | Long.class
 * StringType         | String.class
 * UnionType          | TaggedObject.class
 * OptionType         | ValueContainer.class
 * RecordType         | Object[].class
 * MapType            | TreeMap.class
 * VariantType        | Variant.class
 * ArrayType(Boolean) | boolean[].class
 * ArrayType(Byte)    | byte[].class
 * ArrayType(Integer) | int[].class
 * ArrayType(Long)    | long[].class
 * ArrayType(Float)   | float[].class
 * ArrayType(Double)  | double[].class
 * ArrayType(Byte)    | byte[].class
 * ArrayType( T )     | Object[].class
 * 
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class DefaultBindingFactory extends TypeBindingFactory {
	
	/**
	 * Construct a binding factory.
	 */
	public DefaultBindingFactory() {
		super();
	}
	
	/**
	 * Construct a scheme factory that appends constructed bindings to the user given
	 * repository  
	 * 
	 * @param repository repository where bindings are placed
	 */
	public DefaultBindingFactory(Map<Datatype, Binding> repository) {
		super(repository);
	}

	@Override
	protected Binding doConstruct(Datatype type)
			throws BindingConstructionException {

		// Exact, non-annotated types
		if (type.equals( Bindings.BOOLEAN.type() )) return Bindings.BOOLEAN;
		if (type.equals( Bindings.BYTE.type() )) return Bindings.BYTE;
		if (type.equals( Bindings.INTEGER.type() )) return Bindings.INTEGER;
		if (type.equals( Bindings.LONG.type() )) return Bindings.LONG;
		if (type.equals( Bindings.FLOAT.type() )) return Bindings.FLOAT;
		if (type.equals( Bindings.DOUBLE.type() )) return Bindings.DOUBLE;
		if (type.equals( Bindings.STRING.type() )) return Bindings.STRING;
		if (type.equals( Bindings.VARIANT.type() )) return Bindings.VARIANT;
		if (type.equals( Bindings.BOOLEAN_ARRAY.type() )) return Bindings.BOOLEAN_ARRAY;
		if (type.equals( Bindings.BYTE_ARRAY.type() )) return Bindings.BYTE_ARRAY;
		if (type.equals( Bindings.INT_ARRAY.type() )) return Bindings.INT_ARRAY;
		if (type.equals( Bindings.LONG_ARRAY.type() )) return Bindings.LONG_ARRAY;
		if (type.equals( Bindings.FLOAT_ARRAY.type() )) return Bindings.FLOAT_ARRAY;
		if (type.equals( Bindings.DOUBLE_ARRAY.type() )) return Bindings.DOUBLE_ARRAY;
		if (type.equals( Bindings.BOOLEAN_ARRAY.type() )) return Bindings.BOOLEAN_ARRAY;
		if (type.equals( Bindings.STRING_ARRAY.type() )) return Bindings.STRING_ARRAY;
		
		// Annotated types 
		if (type instanceof BooleanType) return new BooleanBindingDefault((BooleanType)type);
		if (type instanceof DoubleType) return new DoubleBindingDefault((DoubleType)type);
		if (type instanceof FloatType) return new FloatBindingDefault((FloatType)type);
		if (type instanceof ByteType) return new ByteBindingDefault((ByteType)type);
		if (type instanceof IntegerType) return new IntegerBindingDefault((IntegerType)type);
		if (type instanceof LongType) return new LongBindingDefault((LongType)type);
		if (type instanceof StringType) return new StringBindingDefault((StringType)type);
		
		// Constructed types
		if (type instanceof ArrayType) {
			ArrayType arrayType = (ArrayType) type;
			Datatype componentType = arrayType.componentType();
			
			if (componentType instanceof BooleanType) return BooleanArrayBinding.createFrom(arrayType);
			if (componentType instanceof ByteType) return ByteArrayBinding.createFrom(arrayType);
			if (componentType instanceof IntegerType) return IntArrayBinding.createFrom(arrayType);
			if (componentType instanceof LongType) return LongArrayBinding.createFrom(arrayType);
			if (componentType instanceof FloatType) return FloatArrayBinding.createFrom(arrayType);
			if (componentType instanceof DoubleType) return DoubleArrayBinding.createFrom(arrayType);

			ObjectArrayBinding binding = new ObjectArrayBinding(arrayType, null);
			inprogress.put(type, binding);
			binding.componentBinding = construct( componentType );						
			inprogress.remove(type);
			return binding;
		}
		
		if (type instanceof OptionalType) {
			OptionalType optionalType = (OptionalType) type;
			Datatype componentType = optionalType.componentType;
			ContainerOptionalBinding binding = new ContainerOptionalBinding( optionalType, null );
			inprogress.put(type, binding);
			binding.componentBinding = construct( componentType );
			inprogress.remove(type);
			return binding;
		}
		
		if (type instanceof RecordType) {			
			RecordType recordType = (RecordType) type;
			Binding componentBindings[] = new Binding[ recordType.getComponentCount() ];
			RecordObjectArrayBinding binding = new RecordObjectArrayBinding(recordType, componentBindings);
			inprogress.put(type, binding);
			for (int i=0; i<componentBindings.length; i++) {
				componentBindings[i] = construct( recordType.getComponentType(i) );
			}
			inprogress.remove(type);
			return binding;
		}
		
		if (type instanceof UnionType) {
			UnionType unionType = (UnionType) type;
			Binding componentBindings[] = new Binding[ unionType.components.length ];
			UnionTaggedObjectBinding binding = new UnionTaggedObjectBinding(unionType, componentBindings);
			inprogress.put(type, binding);
			for (int i=0; i<componentBindings.length; i++) {
				componentBindings[i] = construct( unionType.getComponent(i).type );
			}
			inprogress.remove(type);
			return binding;
		}		
		
		if (type instanceof MapType) {
			MapType mapType = (MapType) type;
			DefaultMapBinding binding = new DefaultMapBinding(mapType, null, null);
			inprogress.put(type, binding);
			binding.setKeyBinding( construct(mapType.keyType) );
			binding.setValueBinding( construct(mapType.valueType) );
			inprogress.remove(type);
			return binding;
		}
		
		throw new BindingConstructionException("Unexpected, I don't know how to create binding for "+type);
	}

	@Override
	public boolean supportsType(Datatype type) {
		// unexpected
		if (failures.containsKey(type)) return false;
		return true;
	}
	
}



