/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.classfactory;

import java.util.IdentityHashMap;

import org.simantics.databoard.Datatypes;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.type.MapType;
import org.simantics.databoard.type.OptionalType;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.type.UnionType;
import org.simantics.databoard.type.VariantType;

/**
 * Signature Visitor builds a signature string from a datatype.
 * The argument is StringBuilder.
 * 
 * Signature is construtructed with the following notation.
 * 
 *   s                        StringType
 *   z                        BooleanType
 *   d                        DoubleType
 *   f                        FloatType
 *   i                        IntegerType
 *   b                        ByteType
 *   j                        LongType
 *   R                        Referable RecordType
 *   o*                       OptionalType, * denotes componentType
 *   a*                       ArrayType, * denotes componentType
 *   r**e                     RecordType, ** denotes fields
 *   u                        UnionType, * denotes components
 *   m**                      MapType, ** denotes keyType and valueType
 *   t                        Datatype
 *   
 * For example, The signature of UUID.class is "rjje" 
 * 
 * @author toni.kalajainen
 */
public class SignatureVisitor implements Datatype.Visitor1 {
	
	public StringBuilder sb = new StringBuilder();
	public int hashcode = 0xbcbcbca;
	public IdentityHashMap<Datatype, Boolean> visited = new IdentityHashMap<Datatype, Boolean>();
	
	public static String toSignature(Datatype type)
	{
		SignatureVisitor sv = new SignatureVisitor();
		type.accept( sv, null );
		String sig = sv.sb.toString();
		return sig;
	}	

	@Override
	public void visit(ArrayType b, Object obj) {
		sb.append('a');
		b.componentType.accept(this, obj);
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(BooleanType b, Object obj) {
		sb.append('z');
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(DoubleType b, Object obj) {
		sb.append('d');
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(FloatType b, Object obj) {
		sb.append('f');
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(IntegerType b, Object obj) {
		sb.append('i');
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(ByteType b, Object obj) {
		sb.append('b');
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(LongType b, Object obj) {
		sb.append('j');
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(OptionalType b, Object obj) {
		sb.append('o');
		b.componentType.accept(this, sb);
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(RecordType b, Object obj) {
		if ( wasVisited(b) ) {
			sb.append('R');
			return;
		}
		if ( b.isReferable() ) {
			sb.append('R');
			return;
		}
		
		sb.append('r');
		for (int i=0; i<b.getComponentCount(); i++) {
			b.getComponentType(i).accept(this, sb);
		}
		sb.append('e');

		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
		for (int i=0; i<b.getComponentCount(); i++) {
			hashcode = 13*hashcode + b.getComponent(i).name.hashCode();
		}
	}

	@Override
	public void visit(StringType b, Object obj) {
		sb.append('s');
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(UnionType b, Object obj) {
		if ( b.equals( Datatypes.getDatatypeUnchecked(Datatype.class) )) {
			sb.append("t");
			return;
		}
		if ( wasVisited(b) ) {
			sb.append('U');
			return;
		}
		sb.append('u');
		for (int i=0; i<b.getComponentCount(); i++) {
			b.getComponentType(i).accept(this, sb);
		}
		sb.append('e');
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(VariantType b, Object obj) {
		sb.append('v');
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}

	@Override
	public void visit(MapType b, Object obj) {
		if ( wasVisited(b) ) return;
		sb.append('m');
		b.keyType.accept(this, obj);
		b.valueType.accept(this, obj);
		hashcode = 13*hashcode + b.getClass().getName().hashCode() + 133*b.metadataHashCode();
	}
	
	/**
	 * Visit and return if was visited
	 * @param type
	 * @return
	 */
	boolean wasVisited(Datatype type) {
		Boolean result = visited.put(type, Boolean.TRUE);
		return result == null ? false : result;
	}
}
