/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.graph.impl;

import java.util.concurrent.Semaphore;

import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.content.Labeler.Modifier;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.StringIndexModifier;
import org.simantics.db.layer0.adapter.StringModifier;
import org.simantics.db.layer0.adapter.TObjectIntPair;
import org.simantics.layer0.utils.representation.StringRepresentation2;
import org.simantics.utils.ui.ErrorLogger;

/**
 * Please implement:
 * <ul>
 * <li>{@link #createModifierInput(String)} - constructs an input for
 * {@link org.simantics.db.layer0.adapter.Modifier#modify(WriteGraph, Object)}
 * from the specified label given by the user.
 * <li>{@link #doModify(WriteGraph, String)} - perform the requested modification
 * into the graph.</li>
 * </ul>
 * 
 * <p>
 * Other points of customization:
 * </p>
 * <ul>
 * <li>{@link #getInitialValue(ReadGraph)} - returns the value that should be shown
 * initially when editing. The default implementation just adapts the input to
 * its String representation, but you may want to customize this.</li>
 * <li>{@link #initializeGraphModifier(ReadGraph)} - allows you to perform custom
 * initialization of the modifier which uses the graph</li>
 * <li>{@link #getResourceToModify()} - allows you to customize the way in which
 * the input INodeContext is resolved into a Resource. The default
 * implementation uses the IAdaptable interface of INodeContext to get the
 * Resource.</li>
 * <li>{@link #verifyModification(String)} - allows for last chance denial of
 * the modification after the user has signalled approval of the modification.</li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 * 
 * @param <T> the input class of the used
 *        {@link org.simantics.db.layer0.adapter.Modifier}
 */
public abstract class GraphStringIndexModifier implements Modifier {

    protected NodeContext        context;

    protected Session             session;

    protected int                 index;

    protected String              initialValue;

    protected StringIndexModifier modifier;

    /**
     * Used to synchronize {@link #isValid(String)} and {@link #modify(String)}
     * with the asynchronous modifier fetch.
     */
    protected Semaphore           modifierReady = new Semaphore(0);

    /**
     * If <code>non-null</code>, the modifier could not be fetched, e.g. adapted
     * from the specified INodeContext.
     */
    protected Throwable           modifierFailed;


    /**
     * @param context
     * @param session
     */
    public GraphStringIndexModifier(NodeContext context, Session session, int index) throws DatabaseException {
        this.context = context;
        this.session = session;
        this.index = index;

        final Resource r = getResourceToModify();
        if (r == null)
            throw new IllegalArgumentException("This modifier does not work for INodeContexts that are not adaptable to a Resource. The context input is: " + context.getConstant(BuiltinKeys.INPUT));

        session.syncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph g) throws DatabaseException {
                initialValue = getInitialValue(g);
                GraphStringIndexModifier.this.modifier = g.adapt(r, StringIndexModifier.class);
                initializeGraphModifier(g);
                modifierReady.release();
            }
        });

    }

    /**
     * @param g
     * @return the value that shall be returned by {@link #getValue()}
     */
    protected String getInitialValue(ReadGraph g) throws DatabaseException {
        StringRepresentation2 sr = g.adapt(getResourceToModify(), StringRepresentation2.class);
        String s = sr.get(g, index);
        return s;
    }

    /**
     * Override to perform graph-based initialization actions for this modifier.
     * 
     * @param g the graph handle
     */
    protected void initializeGraphModifier(ReadGraph g) {
    }

    /**
     * @return the Resource to modify based on the input INodeContext. This
     *         resource must be adaptable to a {@link StringModifier} in order
     *         for this modifier to work.
     */
    protected Resource getResourceToModify() {
        Resource r = (Resource) context.getAdapter(Resource.class);
        if (r == null)
            throw new AssertionError("context.getAdapter(Resource.class) returned null");
        return r;
    }

    /**
     * @return the modifier or <code>null</code> if the StringModifier adaption
     *         has not completed yet.
     */
    protected StringIndexModifier getModifier() {
        return modifier;
    }

    @Override
    public String getValue() {
        return initialValue;
    }

    @Override
    public String isValid(String label) {
        if (modifierFailed != null)
            return "Could not resolve validator for this value, modification denied. Reason: " + modifierFailed.getMessage();
        if (modifier == null)
            // Cannot validate yet, the validator has not been resolved.
            // For the time being, consider the value invalid for no
            // apparent reason to the user.
            return "";
        TObjectIntPair<String> t = createModifierInput(label);
        return modifier.isValid(t);
    }

    @Override
    public final void modify(String label) {
        if (modifier == null) {
            try {
                modifierReady.acquire();
            } catch (InterruptedException e) {
                // TODO: throw exception?
                return;
            }
        }
        if (modifierFailed != null)
            // TODO: throw exception?
            return;
        final TObjectIntPair<String> t = createModifierInput(label);
        if (!verifyModification(t))
            return;
        session.asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                doModify(graph, t);
            }
        }, parameter -> {
            if (parameter != null)
                ErrorLogger.defaultLogError(parameter);
        });
    }

    /**
     * Called one last time before actually performing the modifying write
     * transaction to verify whether this is really desired or not.
     * 
     * <p>
     * This default implementation will always allow the modification to proceed.
     * </p>
     * 
     * @param label the label to be given to the modifier
     * @return <code>true</code> to go forward with the transaction,
     *         <code>false</code> to bail out
     */
    protected boolean verifyModification(TObjectIntPair<String> label) {
        return true;
    }

    public abstract void doModify(WriteGraph graph, TObjectIntPair<String> label) throws DatabaseException;

    /**
     * Constructs T from the specified label which is then given to
     * {@link #doModify(WriteGraph, Object)} as the class T argument.
     * 
     * @param fromLabel the modified label specified by the user
     * @return the
     *         {@link org.simantics.db.layer0.adapter.Modifier#modify(WriteGraph, Object)}
     *         input
     */
    public abstract TObjectIntPair<String> createModifierInput(String fromLabel);

};
