/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.annotation.ui.wizard;

import java.io.File;
import java.util.Collections;
import java.util.List;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class AnnotationTypeImportPage extends WizardPage {

    /**
     * If non-null, the wizard cannot continue. This message tells why.
     */
    String              failure;

    ImportPlan          importModel;
    Text                importTarget;
    CCombo              importLocation;

    List<NamedResource> models = Collections.emptyList();
    Button              overwrite;

    protected AnnotationTypeImportPage(ImportPlan model) {
        super("Import Annotation Type", "Define Import Location", null);
        this.importModel = model;
    }

    @Override
    public void createControl(Composite parent) {
        Composite container = new Composite(parent, SWT.NONE);
        {
            GridLayout layout = new GridLayout();
            layout.horizontalSpacing = 20;
            layout.verticalSpacing = 10;
            layout.numColumns = 3;
            container.setLayout(layout);
        }

        new Label(container, SWT.NONE).setText("Import target:");
        importTarget = new Text(container, SWT.BORDER);
        {
            importTarget.setEditable(false);
            importTarget.setText("");
            importTarget.setToolTipText("Shows the target of the import.");
            GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(importTarget);
        }

        new Label(container, SWT.NONE).setText("&Annotation type file:");
        importLocation = new CCombo(container, SWT.BORDER);
        {
            importLocation.setText("");
            GridDataFactory.fillDefaults().grab(true, false).span(1, 1).applyTo(importLocation);
            importLocation.addModifyListener(new ModifyListener(){
                @Override
                public void modifyText(ModifyEvent e) {
                    validatePage();
                }
            });
        }
        Button browseFileButton = new Button(container, SWT.PUSH);
        {
            browseFileButton.setText("Br&owse...");
            browseFileButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
            browseFileButton.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    FileDialog dialog = new FileDialog(getShell(), SWT.OPEN);
                    dialog.setText("Choose Annotation Type to Import");
                    String loc = importLocation.getText();
                    dialog.setFilterPath(loc);
                    dialog.setFilterExtensions(new String[] { "*.annotationType" });
                    dialog.setFilterNames(new String[] { "Annotation Type (*.annotationType)" });
                    String file = dialog.open();
                    if (file == null)
                        return;
                    importLocation.setText(file);
                    validatePage();
                }
            });
        }

        try {
            initializeData();
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
        }

        setControl(container);
        validatePage();
    }

    private void initializeData() throws DatabaseException {
        NamedResource target = importModel.sessionContext.getSession().syncRequest(new Read<NamedResource>() {
            @Override
            public NamedResource perform(ReadGraph graph) throws DatabaseException {
                Layer0 L0 = Layer0.getInstance(graph);
                ModelingResources MOD = ModelingResources.getInstance(graph);

                for (Resource r : ISelectionUtils.getPossibleKeys(importModel.selection, SelectionHints.KEY_MAIN, Resource.class)) {
                	return toNamedResource(graph, r);
                }
                for (Resource r : graph.sync(new ObjectsWithType(Simantics.getProjectResource(), L0.ConsistsOf, MOD.StructuralModel))) {
                	return toNamedResource(graph, r);
                }
                return null;
            }

            private NamedResource toNamedResource(ReadGraph graph, Resource lib) throws DatabaseException {
                Resource model = graph.sync(new PossibleModel(lib));
                if (model == null)
                    return new NamedResource(NameUtils.getSafeName(graph, lib), lib);
                return new NamedResource(NameUtils.getSafeName(graph, model) + ": " + NameUtils.getSafeName(graph, lib), lib);
            }
        });

        if (target == null) {
            failure = "No models found in the database.";
            return;
        }

        importTarget.setText(target.getName());
        importModel.selection = target.getResource();

        for (String path : importModel.recentLocations) {
            importLocation.add(path);
        }
        if (importLocation.getItemCount() > 0)
            importLocation.select(0);
    }

    void validatePage() {
        if (failure != null) {
            setErrorMessage(failure);
            setPageComplete(false);
            return;
        }
        String importLoc = importLocation.getText();
        if (importLoc.isEmpty()) {
            setMessage("Select file to import.");
            setErrorMessage(null);
            setPageComplete(false);
            return;
        }
        File file = new File(importLoc);
        if (!file.exists() || !file.isFile()) {
            setErrorMessage("Selected file is invalid.");
            setPageComplete(false);
            return;
        }
        importModel.importLocation = file;

        setErrorMessage(null);
        setMessage("Import " + file.getName() + "");
        setPageComplete(true);
    }

}
