package org.simantics.structural2.scl;

import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.structural2.internal.Activator;

/**
 * @author Tuukka Lehtonen
 * @since 1.51.0
 */
public interface ComponentTypeScriptPhaseContribution {

	public static class Phase {
		public final String id;
		public final String label;

		public Phase(String id, String label) {
			this.id = id;
			this.label = label;
		}
	}

	/**
	 * Get available SCL script execution phases from this contribution for the
	 * specified input script and component type pair.
	 * 
	 * @param graph
	 * @param input
	 * @return provided execution phases or <code>null</code> if this contribution
	 *         doesn't provide anything for the specified input
	 * @throws DatabaseException
	 */
	Phase[] getPhases(ReadGraph graph, Resource script, Resource componentType) throws DatabaseException;

	/**
	 * Consult all existing {@link ComponentTypeScriptPhaseContribution} OSGi
	 * services to see if any of them would provide execution phases for the input
	 * script.
	 * 
	 * @param graph
	 * @param script
	 * @return the first provided execution phases or <code>null</code> if nothing
	 *         was provided by any contribution
	 * @throws DatabaseException
	 */
	public static Phase[] getPhasesForScript(ReadGraph graph, Resource script) throws DatabaseException {
		StructuralResource2 STR = StructuralResource2.getInstance(graph);
		Resource componentType = graph.getPossibleObject(script, STR.ComponentType_hasScript_Inverse);

		BundleContext ctx = Activator.getContext();
		try {
			ServiceReference<?>[] refs = ctx.getAllServiceReferences(ComponentTypeScriptPhaseContribution.class.getName(), null);
			if (refs != null) {
				for (ServiceReference<?> ref : refs) {
					ComponentTypeScriptPhaseContribution c = (ComponentTypeScriptPhaseContribution) ctx.getService(ref);
					try {
						Phase[] phases = c.getPhases(graph, script, componentType);
						if (phases != null) {
							return phases;
						}
					} finally {
						ctx.ungetService(ref);
					}
				}
			}
			return null;
		} catch (InvalidSyntaxException | SecurityException | IllegalStateException | IllegalArgumentException e) {
			//LoggerFactory.getLogger(ComponentTypeScriptPhaseContribution.class).error("Failed to get ComponentTypeScriptPhaseContribution services", e);
			throw new DatabaseException(e);
		}
	}

	public static Phase[] getPhasesForScript(RequestProcessor proc, Resource script) throws DatabaseException {
		return proc.syncRequest((ReadGraph graph) -> getPhasesForScript(graph, script));
	}

}