package org.simantics.scl.compiler.markdown.html;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import org.simantics.scl.compiler.module.repository.ModuleRepository;

public class GenerateAllHtmlDocumentation {
    public static void generate(ModuleRepository moduleRepository, Path directory) throws IOException {
        HierarchicalDocumentationRef root = HierarchicalDocumentationRef.generateTree(moduleRepository.getSourceRepository());
        generate(moduleRepository, directory, root.getChildren(), 0, root);
        generateIndex(directory, 0, "StandardLibrary/Prelude.html");
    }

    private static void generateIndex(Path directory, int level, String url) throws IOException {
        Path documentationFile = directory.resolve("index.html");
        for(int i=0;i<level;++i)
            url = "../" + url;
        String content = "<html><head><meta http-equiv=\"refresh\" content=\"1;url="+url+"\">" +
                         "<script type=\"text/javascript\">window.location.href = \""+url+"\"</script>" +
                         "</head><body>If you are not redirected automatically, <a href=\""+url+"\">follow the link</a>.</body></html>";
        Files.write(documentationFile, content.getBytes());
    }

    private static void generate(ModuleRepository moduleRepository, Path directory, List<HierarchicalDocumentationRef> refs, int level, HierarchicalDocumentationRef root) throws IOException {
        String navigation = generateNavigation(level, root);
        for(HierarchicalDocumentationRef ref : refs) {
            String name = ref.getName();
            
            // Documentation for this ref
            String documentationName = ref.getDocumentationName();
            if(documentationName != null) {
                String documentation = HtmlDocumentationGeneration.generate(moduleRepository, documentationName, navigation);
                Path documentationFile = directory.resolve(name + ".html"); 
                Files.write(documentationFile, documentation.getBytes());
            }
            
            // Children of the ref
            List<HierarchicalDocumentationRef> children = ref.getChildren();
            if(!children.isEmpty()) {
                Path childDirectory = directory.resolve(name);
                if(!Files.exists(childDirectory))
                    Files.createDirectory(childDirectory);
                generate(moduleRepository, childDirectory, children, level+1, root);
                if(documentationName != null)
                    generateIndex(childDirectory, 1, ref.getName() + ".html");
                else
                    generateIndex(childDirectory, level+1, "StandardLibrary/Prelude.html");
            }
        }
    }

    private static String generateNavigation(int level,
            HierarchicalDocumentationRef root) {
        StringBuilder b = new StringBuilder();
        generateNavigation(b, level, root.getChildren());
        return b.toString();
    }

    private static void generateNavigation(StringBuilder b, int level,
            List<HierarchicalDocumentationRef> refs) {
        b.append("<ul>");
        for(HierarchicalDocumentationRef ref : refs) {
            b.append("<li>");
            String documentationName = ref.getDocumentationName();
            if(documentationName != null) {
                if(!documentationName.contains("/"))
                    documentationName = "StandardLibrary/" + documentationName;
                b.append("<a href=\"");
                for(int i=0;i<level;++i)
                    b.append("../");
                b.append(documentationName);
                b.append(".html\">");
            }
            b.append(ref.getName());
            if(documentationName != null)
                b.append("</a>");
            List<HierarchicalDocumentationRef> children = ref.getChildren();
            if(children != null)
                generateNavigation(b, level, children);
            b.append("</li>");
        }
        b.append("</ul>");
    }
}
