package org.simantics.scl.compiler.elaboration.expressions.annotations;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.simantics.scl.compiler.constants.StringConstant;
import org.simantics.scl.compiler.elaboration.expressions.EListLiteral;
import org.simantics.scl.compiler.elaboration.expressions.ELiteral;
import org.simantics.scl.compiler.elaboration.expressions.EStringLiteral;
import org.simantics.scl.compiler.elaboration.expressions.EVar;
import org.simantics.scl.compiler.elaboration.expressions.Expression;
import org.simantics.scl.compiler.errors.ErrorLog;
import org.simantics.scl.compiler.internal.parsing.declarations.DAnnotationAst;

public class AnnotationUtils {
    /**
     * Processes an annotation of form
     * <pre>&#64;Annotation "text"</pre>
     * or
     * <pre>&#64;Annotation text</pre>
     */
    public static String processStringAnnotation(ErrorLog errorLog, DAnnotationAst annotation) {
        if(annotation.parameters.length != 1)
            errorLog.log(annotation.location, "Expected one string parameter for " + annotation.id.text);
        String result = extractString(annotation.parameters[0]);
        if(result == null)
            errorLog.log(annotation.location, "Expected a string parameter for " + annotation.id.text);
        return result;
    }
    
    public static void processTagAnnotation(ErrorLog errorLog, DAnnotationAst annotation) {
        if(annotation.parameters.length != 0)
            errorLog.log(annotation.location, "Expected no parameters for " + annotation.id.text);
    }
    
    public static String extractString(Expression expression) {
        if(expression instanceof EVar)
            return ((EVar)expression).name;
        else if(expression instanceof EStringLiteral) {
            EStringLiteral literal = (EStringLiteral)expression;
            if(literal.strings.length == 1)
                return literal.strings[0];
        }
        else if(expression instanceof ELiteral) {
            ELiteral literal = (ELiteral)expression;
            if(literal.getValue() instanceof StringConstant) {
                StringConstant constant = (StringConstant)literal.getValue();
                return constant.getValue();
            }
        }
        return null;
    }
    
    public static List<EVar> extractIdentifierList(Expression expression) {
        if(expression instanceof EVar || expression instanceof EStringLiteral || expression instanceof ELiteral)
            return Collections.singletonList(new EVar(expression.location, extractString(expression)));
        else if(expression instanceof EListLiteral) {
            Expression[] components = ((EListLiteral)expression).getComponents();
            EVar[] items = new EVar[components.length];
            for(int i=0;i<components.length;++i) {
                String value = extractString(components[i]);
                if(value == null)
                    return null;
                items[i] = new EVar(components[i].location, value);
            }
            return Arrays.asList(items);
        }
        return null;
    }
}
