package org.simantics.export.core.intf;

import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.osgi.service.prefs.Preferences;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.type.RecordType;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.error.ExportException;
import org.simantics.export.core.manager.Content;

/**
 * Interface for publisher implementations.
 *
 * @author toni.kalajainen@semantum.fi
 */
public interface PublisherClass {
	
	/**
	 * Publish content
	 * 
	 * @param ctx
	 * @param contents
	 * @param options all options 
	 * @param location location settings
	 * @param progress (Optional) progress monitor
	 * @throws ExportException 
	 */
	void publish( ExportContext ctx, List<Content> contents, Variant options, Variant location, IProgressMonitor progress ) throws ExportException;
	
	/**
	 * Validate before publish 
	 * 
	 * @param ctx
	 * @param contents
	 * @param options overall all options
	 * @param location location settings
	 * @return
	 * @throws ExportException 
	 */
	List<String> validate( ExportContext ctx, List<Content> contents, Variant options, Variant location ) throws ExportException;
	
	//// Location Management ////
	
	/**
	 * Get all location options (even the ones that are hidden from the end-user)  
	 * 
	 * @param ctx
	 * @param contents contents to publish
	 * @param options all options
	 * @return
	 * @throws ExportException
	 */
	RecordType locationOptions( ExportContext ctx, List<Content> contents ) throws ExportException;
	
	/**
	 * Create or edit location. This may cause blocking UI operation.
	 * 
	 * @param ctx
	 * @param location Some location settings
	 * @return new or modified location
	 * @throws ExportException
	 */
	Variant createLocation( ExportContext ctx, Variant location ) throws ExportException;
	
	/**
	 * Checks if location exists.
	 * 
	 * @param ctx
	 * @param location
	 * @return true if location exists
	 * @throws ExportException
	 */
	boolean locationExists( ExportContext ctx, Variant location ) throws ExportException;
	
	/**
	 * Fill default location values to location
	 * 
	 * @param ctx
	 * @param contents
	 * @param options
	 * @param locationOptions
	 * @throws ExportException
	 */
	void fillDefaultPrefs( ExportContext ctx, List<Content> contents, Variant options, Variant locationOptions ) throws ExportException;
	
	/**
	 * Save location options to preferences node(s).
	 * 
	 * @param locationOptions
	 * @param contentScopePrefs
	 * @param workspaceScopePrefs
	 * @throws ExportException
	 */
	void savePref( Variant locationOptions, Preferences contentScopePrefs, Preferences workspaceScopePrefs ) throws ExportException;
	
	/**
	 * Load location options from preferences node(s).
	 * 
	 * @param locationOptions
	 * @param contentScopePrefs
	 * @param workspaceScopePrefs
	 * @throws ExportException
	 */
	void loadPref( Variant locationOptions, Preferences contentScopePrefs, Preferences workspaceScopePrefs ) throws ExportException;

}
