package org.simantics.document.swt.core.selection;

import java.util.function.Consumer;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.common.views.IFilterArea;
import org.simantics.browsing.ui.common.views.IFilterAreaProvider;
import org.simantics.browsing.ui.swt.PartNameListener;
import org.simantics.db.ReadGraph;
import org.simantics.db.common.procedure.adapter.ListenerSupport;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.request.Read;
import org.simantics.document.swt.core.SWTDocumentClient;
import org.simantics.document.swt.core.SWTViews;
import org.simantics.selectionview.IPropertyTab;
import org.simantics.selectionview.IPropertyTab2;
import org.simantics.selectionview.PropertyTabContributor;
import org.simantics.utils.ui.ISelectionUtils;
import org.simantics.utils.ui.jface.ActiveSelectionProvider;

final public class DocumentTabContributor implements PropertyTabContributor, ListenerSupport {

    // For ListenerSupport (supporting DB request listeners)
    protected boolean    disposed = false;
    protected ISelection input    = StructuredSelection.EMPTY;
    final protected String documentURI;
    private SWTDocumentClient client;
    
    protected ISelectionProvider selectionProvider = new ActiveSelectionProvider();
    
    public DocumentTabContributor(String documentURI) {
    	this.documentURI = documentURI;
    }

    public void createControls(Composite body, IWorkbenchSite site) {
    	
    	client = new SWTDocumentClient(SWTViews.getMapping(), selectionProvider, site, body);
    	
    }

    public IFilterArea getFilterArea() {
        return null;
    }

    public void requestFocus() {
    }

    public ISelectionProvider getSelectionProvider() {
        return selectionProvider; 
    }

    public Read<String> getPartNameReadRequest(ISelection forSelection) {
        return null;
    }

    public void updatePartName(ISelection forSelection, Consumer<String> updateCallback) {
        Read<String> read = getPartNameReadRequest(forSelection);
        if (read == null) {
            updateCallback.accept("Selection");
        } else {
            Simantics.getSession().asyncRequest(read, new PartNameListener(updateCallback, this));
        }
    }

    public void updatePartName(Consumer<String> updateCallback) {
        updatePartName(input, updateCallback);
    }

    protected void dispose() {
        this.disposed = true;
    }

    @Override
    public void exception(Throwable t) {
        ErrorLogger.defaultLogError("PropertyTabContributorImpl received unexpected exception.", t);
    }

    @Override
    public boolean isDisposed() {
        return disposed;
    }

    public void createControl(Composite parent, final IWorkbenchSite site) {

        class TabComposite extends Composite {
            public TabComposite(Composite parent) {
                super(parent, 0);

                GridLayoutFactory.fillDefaults().applyTo(parent);
                GridDataFactory.fillDefaults().span(1, 1).grab(true, true).applyTo(this);

                Composite body = this;
                GridLayoutFactory.fillDefaults().spacing(0, 0).equalWidth(false).numColumns(1).applyTo(body);

                try {
                	DocumentTabContributor.this.createControls(body, site);
                } catch (Throwable t) {
                    ErrorLogger.defaultLogError(t);
                }
            }
        }

        final TabComposite tc = new TabComposite(parent);
        tc.addListener(SWT.Dispose, new Listener() {
            public void handleEvent(Event e) {
            	DocumentTabContributor.this.dispose();
            }
        });
    }

    @Override
    public IPropertyTab create(Composite parent, IWorkbenchSite site, ISessionContext context, Object input) {
        IPropertyTab tab = new Tab(site, parent);
        tab.createControl((Composite) tab.getControl(), context);
        return tab;
    }

    class Tab extends Composite implements IPropertyTab2, IFilterAreaProvider {

        final IWorkbenchSite    site;
        
        public Tab(IWorkbenchSite site, Composite parent) {
            super(parent, SWT.NONE);
            this.site = site;
        }

        @Override
        public void createControl(Composite parent, ISessionContext context) {
        	DocumentTabContributor.this.createControl(parent, site);
        }

        @Override
        public Control getControl() {
            return this;
        }

        @Override
        public boolean isDisposed() {
            return super.isDisposed();
        }

        @Override
        public void requestFocus() {
        	DocumentTabContributor.this.requestFocus();
        }

        @Override
        public void setInput(ISessionContext context, ISelection selection, boolean force) {
        	
        	DocumentTabContributor.this.input = selection;

			try {
				
				String input = Simantics.getSession().syncRequest(new UniqueRead<String>() {

					@Override
					public String perform(ReadGraph graph) throws DatabaseException {
						Variable var = ISelectionUtils.filterSingleSelection(DocumentTabContributor.this.input, Variable.class);
						if(var == null) return null;
						return var.getURI(graph).substring("http:/".length());
					}

				});
				
				ListenerSupport support = new ListenerSupport() {

					@Override
					public void exception(Throwable t) {
						Logger.defaultLogError(t);
					}

					@Override
					public boolean isDisposed() {
						return DocumentTabContributor.this.isDisposed();
					}
					
				};
				
				if(input != null) {
					client.track(support, documentURI, input);
					layout(true, true);
				}
	        	
			} catch (DatabaseException e) {
				Logger.defaultLogError(e);
			}
            
        }

        @Override
        public ISelectionProvider getSelectionProvider() {
            return DocumentTabContributor.this.getSelectionProvider();
        }

        @Override
        public IFilterArea getFilterArea() {
            return DocumentTabContributor.this.getFilterArea();
        }

        @Override
        public void updatePartName(Consumer<String> updateCallback) {
        	DocumentTabContributor.this.updatePartName(input, updateCallback);
        }

    }
    
    @Override
    public int hashCode() {
    	return documentURI.hashCode();
    }
    
    @Override
    public boolean equals(Object object) {
    	
        if (this == object)
            return true;
        else if (object == null)
            return false;
        else if (!(object instanceof DocumentTabContributor))
            return false;

        DocumentTabContributor dtc = (DocumentTabContributor)object;
        return documentURI.equals(dtc.documentURI);
        
    }
	
}
