/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UnaryAsyncRead;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.diagram.connection.ConnectionSegmentEnd;
import org.simantics.diagram.connection.ConnectionSegmentType;
import org.simantics.diagram.content.EdgeResource;
import org.simantics.diagram.stubs.DiagramResource;

/**
 * @author Antti Villberg
 *
 */
public class ConnectionInfoRequest extends UnaryAsyncRead<EdgeResource, ConnectionInfo> {

    static class SingleConnectionInfo {
        public Resource connection;
        public ConnectionSegmentEnd end;
    }

    final DiagramResource dr;

    private SingleConnectionInfo first;
    private SingleConnectionInfo second;

    public ConnectionInfoRequest(DiagramResource dr, EdgeResource resource) {
        super(resource);
        this.dr = dr;
    }

    public void fillPossibleType(AsyncReadGraph graph, Resource resource, final AsyncProcedure<ConnectionInfo> procedure, final ConnectionInfo result) {
        if (resource == null) {
            procedure.execute(graph, result);
        } else {
            graph.forSingleType(resource, dr.Connection, new AsyncProcedure<Resource>() {
                @Override
                public void exception(AsyncReadGraph graph, Throwable throwable) {
                    procedure.exception(graph, throwable);
                }

                @Override
                public void execute(AsyncReadGraph graph, Resource type) {
                    result.connectionType = type;
                    procedure.execute(graph, result);
                }
            });
        }
    }

    ConnectionInfo createConnectionInfo(Resource connection) {
        ConnectionInfo info = new ConnectionInfo();
        info.connection = connection;
        info.firstEnd = first.end;
        info.secondEnd = second.end;
        info.segmentType = ConnectionSegmentType.toSegmentType(first.end, second.end);
        return info;
    }

    public void isBranchPointOf(AsyncReadGraph graph, Resource resource, final AsyncProcedure<ConnectionInfo> procedure, final SingleConnectionInfo result, final AtomicInteger ready) {

        graph.forPossibleObject(resource, dr.IsBranchPointOf, new AsyncProcedure<Resource>() {
            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                procedure.exception(graph, throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, Resource connection) {
                result.end = ConnectionSegmentEnd.BRANCH;
                result.connection = connection;
                if (ready.decrementAndGet() == 0) {
                    ConnectionInfo info = createConnectionInfo(connection);
                    fillPossibleType(graph, connection, procedure, info);
                }
            }
        });

    }

    public void isConnectorOf(AsyncReadGraph graph, final Resource resource, final AsyncProcedure<ConnectionInfo> procedure, final SingleConnectionInfo result, final AtomicInteger ready) {

        graph.forPossibleObject(resource, dr.IsConnectorOf, new AsyncProcedure<Resource>() {
            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                procedure.exception(graph, throwable);
            }

            @Override
            public void execute(AsyncReadGraph graph, Resource connection) {
                if (connection != null) {
                    result.end = ConnectionSegmentEnd.CONNECTOR;
                    result.connection = connection;
                    if (ready.decrementAndGet() == 0) {
                        ConnectionInfo info = createConnectionInfo(connection);
                        fillPossibleType(graph, connection, procedure, info);
                    }
                } else {
                    isBranchPointOf(graph, resource, procedure, result, ready);
                }
            }
        });

    }

    @Override
    public void perform(AsyncReadGraph graph, AsyncProcedure<ConnectionInfo> procedure) {
        AtomicInteger ready = new AtomicInteger(2);
        first = new SingleConnectionInfo();
        second = new SingleConnectionInfo();
        isConnectorOf(graph, parameter.first(), procedure, first, ready);
        isConnectorOf(graph, parameter.second(), procedure, second, ready);
    }

}
