/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.genericrelation.combinators;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.GenericRelation;
import org.simantics.db.layer0.genericrelation.AbstractRelation;
import org.simantics.utils.datastructures.Pair;

/**
 * Takes the union of multiple relations. Does not remove duplicates.
 */
public class Union extends AbstractRelation {

	Collection<GenericRelation> relations;
	
	public Union(Collection<GenericRelation> relations) {
		this.relations = relations;
	}

	@Override
	public boolean contains(ReadGraph g, Object[] tuple) throws DatabaseException {
		for(GenericRelation relation : relations)
			if(relation.contains(g, tuple))
				return true;
		return false;
	}

	@Override
	public boolean isRealizable() {
		for(GenericRelation relation : relations)
			if(!relation.isRealizable())
				return false;
		return true;
	}

	@Override
	public List<Object[]> realize(ReadGraph g) throws DatabaseException {
		ArrayList<List<Object[]>> results = new ArrayList<List<Object[]>>(relations.size());		
		for(GenericRelation relation : relations)
			results.add(relation.realize(g));
		int size = 0;
		for(List<Object[]> part : results)
			size += part.size();
		ArrayList<Object[]> result = new ArrayList<Object[]>(size);
		for(List<Object[]> part : results)
			result.addAll(part);
		return result;
	}
	
	@Override
	public GenericRelation selectByRelation(String bindingPattern, GenericRelation constraint) {
		ArrayList<GenericRelation> semijoinRelations = new ArrayList<GenericRelation>(relations.size()); 
		for(GenericRelation relation : relations)
			semijoinRelations.add(relation.selectByRelation(bindingPattern, constraint));
		return new Union(semijoinRelations);
	}
	
	@Override
	public GenericRelation select(String bindingPattern, Object[] constants) {
		ArrayList<GenericRelation> selectedRelations = new ArrayList<GenericRelation>(relations.size()); 
		for(GenericRelation relation : relations)
			selectedRelations.add(relation.select(bindingPattern, constants));
		return new Union(selectedRelations);
	}

	@Override
	public Pair<String, String>[] getFields() {
		for(GenericRelation relation : relations)
			return relation.getFields();
		return null;
	}

}
