package org.simantics.db.layer0;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.simantics.datatypes.literal.GUID;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.db.layer0.adapter.impl.EntityInstances.QueryIndex;
import org.simantics.db.layer0.genericrelation.IndexQueries;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.layer0.Layer0;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.utils.datastructures.Triple;

public final class QueryIndexUtils {

    public static List<Resource> searchByTypeShallow(ReadGraph graph, Resource model, Resource type) throws DatabaseException {
        return graph.syncRequest(new QueryIndex(model, type, ""), TransientCacheListener.<List<Resource>>instance());
    }

    public static List<Resource> searchByType(ReadGraph graph, Resource model, Resource type) throws DatabaseException {
        Instances query = graph.adapt(type, Instances.class);
        return Layer0Utils.sortByClusterUnique(graph, query.find(graph, model));
    }

    public static List<Resource> searchByGUID(ReadGraph graph, Resource indexRoot, GUID guid) throws DatabaseException {
        return searchByGUID(graph, indexRoot, guid.indexString());
    }
    
    public static List<Resource> searchByGUID(ReadGraph graph, Resource indexRoot, String indexString) throws DatabaseException {
        return searchByQueryShallow(graph, indexRoot, "GUID:" + IndexQueries.quoteTerm(indexString));
    }

    public static List<Resource> searchByQueryShallow(ReadGraph graph, Resource model, String query) throws DatabaseException {
        return graph.syncRequest(new QueryIndex(model, Layer0.getInstance(graph).Entity, query), TransientCacheListener.<List<Resource>>instance());
    }

    public static List<Resource> searchByQuery(ReadGraph graph, Resource model, String query) throws DatabaseException {
        Instances instances = graph.adapt(Layer0.getInstance(graph).Entity, Instances.class);
        Collection<Resource> queryResult = instances.find(graph, model, query);
        return Layer0Utils.sortByClusterUnique(graph, queryResult);
    }

    public static List<Resource> searchByTypeAndFilter(ReadGraph graph, Resource model, Resource type, Function1<Resource,Boolean> filter) throws DatabaseException {
        Instances query = graph.adapt(type, Instances.class);
        ArrayList<Resource> result =  new ArrayList<>();
        for(Resource r : query.find(graph, model)) {
            if(filter.apply(r))
                result.add(r);
        }
        return result;
    }

    public static List<Triple<Resource, Resource, String>> getIndexEntries(ReadGraph graph, Resource model, String filter) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        List<Resource> entries = searchByQuery(graph, model, filter);
        List<Triple<Resource, Resource, String>> listOfTriples = new ArrayList<Triple<Resource,Resource,String>>();
        for (Resource entry : entries) {
            Resource type = graph.getPossibleObject(entry, L0.InstanceOf);
            String name = NameUtils.getSafeName(graph, entry);
            listOfTriples.add(new Triple<>(entry, type, name));
        }
        return listOfTriples;
    }

    public static String listIndexEntries(ReadGraph graph, Resource model, String filter) throws DatabaseException {
        List<Triple<Resource, Resource, String>> listOfTriples = getIndexEntries(graph, model, filter);
        StringBuilder sb = new StringBuilder();
        sb.append("== LISTING INDEX ENTRIES OF INDEX: " + NameUtils.getSafeName(graph, model) + ". AMOUNT OF ENTRIES: " + listOfTriples.size() + " ==\n");
        for (Triple<Resource, Resource, String> entry : listOfTriples) {
            String instanceOf = NameUtils.getSafeName(graph, entry.second);
            sb.append("Name: " + entry.third + " instanceOf: " + instanceOf + " Resource: " + entry.first.toString() + "\n");
        }
        return sb.toString();
    }

    public static List<Resource> searchByTypeAndName(ReadGraph graph, Resource model, Resource type, String name) throws DatabaseException {
        Instances query = graph.adapt(type, Instances.class);
        ArrayList<Resource> result =  new ArrayList<>();
        for(Resource r : query.findByName(graph, model, name)) {
            if(graph.isInstanceOf(r, type))
                result.add(r);
        }
        return result;
    }

    public static List<Resource> searchByTypeAndNameShallow(ReadGraph graph, Resource model, Resource type, String name) throws DatabaseException {
        return graph.syncRequest(new QueryIndex(model, type, IndexQueries.quoteTerm(name)), TransientCacheListener.<List<Resource>>instance());
    }
}
