/*******************************************************************************
 * Copyright (c) 2007, 2021 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy
 *******************************************************************************/
package org.simantics.browsing.ui;

import java.util.Collection;

import org.simantics.browsing.ui.NodeContext.ConstantKey;
import org.simantics.browsing.ui.NodeContext.ParametrizedPrimitiveQueryKey;
import org.simantics.browsing.ui.NodeContext.PrimitiveQueryKey;
import org.simantics.browsing.ui.NodeContext.QueryKey;
import org.simantics.browsing.ui.content.CheckedStateFactory;
import org.simantics.browsing.ui.content.ComparableContext;
import org.simantics.browsing.ui.content.ComparableContextFactory;
import org.simantics.browsing.ui.content.ImageDecorator;
import org.simantics.browsing.ui.content.ImageDecoratorFactory;
import org.simantics.browsing.ui.content.Imager;
import org.simantics.browsing.ui.content.ImagerFactory;
import org.simantics.browsing.ui.content.LabelDecorator;
import org.simantics.browsing.ui.content.LabelDecoratorFactory;
import org.simantics.browsing.ui.content.Labeler;
import org.simantics.browsing.ui.content.LabelerFactory;
import org.simantics.browsing.ui.content.PrunedChildrenResult;
import org.simantics.browsing.ui.content.Viewpoint;
import org.simantics.browsing.ui.content.ViewpointContribution;
import org.simantics.browsing.ui.content.ViewpointContributionFactory;
import org.simantics.browsing.ui.content.ViewpointFactory;

/**
 * Keys and key classes for different queries and graph explorer nodes.
 * 
 * <p>
 * <code>QueryKey</code> instances are for identifying normal queries to be
 * performed. Also {@link NodeQueryProcessor}s, i.e. the query implementations,
 * use them to identify the query that they are capable of answering (see
 * {@link NodeQueryProcessor#getIdentifier()} method).
 * </p>
 * 
 * <p>
 * Both <code>PrimitiveQueryKey</code> instances and
 * <code>ParametrizedPrimitiveQueryKey</code> classes are used for identifying
 * queries that are performed by {@link PrimitiveQueryProcessor}s.
 * </p>
 * 
 * TODO: specify for every query key or query key class: description, query dependencies, query return value
 */
public interface BuiltinKeys {

    public static class InputKey implements ConstantKey<Object> {
        private InputKey() {}
        @Override
        public String toString() {
            return "INPUT"; //$NON-NLS-1$
        }
    };

    /**
     * This key is used for storing the main input object into a
     * {@link NodeContext}.
     * 
     * <p>
     * See <code>org.simantics.browsing.ui.common.NodeContextBuilder</code> and
     * <code>org.simantics.browsing.ui.common.NodeContextUtil</code> for ways of
     * creating NodeContext instances.
     */
    public static final ConstantKey<Object> INPUT = new InputKey();

    public static class UIContextKey implements ConstantKey<String> {
        private UIContextKey() {}
        @Override
        public String toString() {
            return "UI_CONTEXT"; //$NON-NLS-1$
        }
    };

    /**
     * This key is used for storing the node-specific ui context into a
     * {@link NodeContext}.
     * 
     * <p>
     * See <code>org.simantics.browsing.ui.common.NodeContextBuilder</code> and
     * <code>org.simantics.browsing.ui.common.NodeContextUtil</code> for ways of
     * creating NodeContext instances.
     */
    public static final ConstantKey<String> UI_CONTEXT = new UIContextKey();
    
    public static class BrowseContextKey implements ConstantKey<Object> {
        private BrowseContextKey() {}
        @Override
        public String toString() {
            return "BROWSE_CONTEXT"; //$NON-NLS-1$
        }
    };

    /**
     * This key is used for storing the node-specific BrowseContext input object into a
     * {@link NodeContext}.
     * 
     * <p>
     * See <code>org.simantics.browsing.ui.common.NodeContextBuilder</code> and
     * <code>org.simantics.browsing.ui.common.NodeContextUtil</code> for ways of
     * creating NodeContext instances.
     */
    public static final ConstantKey<Object> BROWSE_CONTEXT = new BrowseContextKey();

    public static class ActionBrowseContextKey implements ConstantKey<Object> {
        private ActionBrowseContextKey() {}
        @Override
        public String toString() {
            return "ACTION_BROWSE_CONTEXT"; //$NON-NLS-1$
        }
    };

    /**
     * This key is used for storing the node-specific ActionBrowseContext input
     * object into a {@link NodeContext}.
     * 
     * <p>
     * See <code>org.simantics.browsing.ui.common.NodeContextBuilder</code> and
     * <code>org.simantics.browsing.ui.common.NodeContextUtil</code> for ways of
     * creating NodeContext instances.
     */
    public static final ConstantKey<Object> ACTION_BROWSE_CONTEXT = new ActionBrowseContextKey();

    public static class FilterKey implements ConstantKey<String> {
        private FilterKey() {}
        @Override
        public String toString() {
            return "FILTER"; //$NON-NLS-1$
        }
    };

    /**
     * This key is used for storing the filter string object into it a
     * {@link NodeContext}. The value is used by primitive query viewpoints to
     * perform pruning.
     * 
     * <p>
     * See <code>org.simantics.browsing.ui.common.NodeContextBuilder</code> and
     * <code>org.simantics.browsing.ui.common.NodeContextUtil</code> for ways of
     * creating NodeContext instances.
     */
    public static final ConstantKey<String> FILTER = new FilterKey();

    public static final QueryKey<Viewpoint> SELECTED_VIEWPOINT = new QueryKey<Viewpoint>() {
        @Override
        public String toString() {
            return "SELECTED_VIEWPOINT"; //$NON-NLS-1$
        }
    };

    public static final QueryKey<String> ACTIVE_FILTER = new QueryKey<String>() {
        @Override
        public String toString() {
            return "ACTIVE_FILTER"; //$NON-NLS-1$
        }
    };

    public static final QueryKey<Collection<ViewpointContribution>> VIEWPOINT_CONTRIBUTIONS = new QueryKey<Collection<ViewpointContribution>>() {
        @Override
        public String toString() {
            return "VIEWPOINT_CONTRIBUTIONS"; //$NON-NLS-1$
        }
    };

    /**
     * A query for all available viewpoint factories for the current input.
     * <p>
     * Depends on: {@link #INPUT}
     * <p>
     * Returns: a non-null collection
     */
    public static final QueryKey<Collection<ViewpointFactory>> VIEWPOINT_FACTORIES = new QueryKey<Collection<ViewpointFactory>>() {
        @Override
        public String toString() {
            return "VIEWPOINT_FACTORIES"; //$NON-NLS-1$
        }
    };

    public static class SelectedViewpointFactoryKey extends ParametrizedPrimitiveQueryKey<ViewpointFactory> {
        public SelectedViewpointFactoryKey(Collection<ViewpointFactory> factories) {
            super(factories);
            assert factories != null;
        }
        @Override
        public String getKeyName() {
            return "SELECTED_VIEWPOINT_FACTORY"; //$NON-NLS-1$
        }
    };

    public static class ViewpointKey extends ParametrizedPrimitiveQueryKey<Viewpoint> {
        public ViewpointKey(ViewpointFactory factory) {
            super(factory);
            assert factory != null;
        }
        @Override
        public String getKeyName() {
            return "VIEWPOINT"; //$NON-NLS-1$
        }
    };

    public static class ViewpointContributionKey extends ParametrizedPrimitiveQueryKey<ViewpointContribution> {
        public ViewpointContributionKey(ViewpointContributionFactory factory) {
            super(factory);
            assert factory != null;
        }
        @Override
        public String getKeyName() {
            return "VIEWPOINT_CONTRIBUTION"; //$NON-NLS-1$
        }
    };

    /**
     * Used by <code>GraphExplorer.setTextAndImage</code>.
     */
    public static final QueryKey<Collection<LabelDecoratorFactory>> LABEL_DECORATOR_FACTORIES = new QueryKey<Collection<LabelDecoratorFactory>>() {
        @Override
        public String toString() {
            return "LABEL_DECORATOR_FACTORIES"; //$NON-NLS-1$
        }
    };
    /**
     * Used by <code>GraphExplorer.setTextAndImage</code>.
     */
    public static final QueryKey<Collection<LabelDecorator>> LABEL_DECORATORS = new QueryKey<Collection<LabelDecorator>>() {
        @Override
        public String toString() {
            return "LABEL_DECORATORS"; //$NON-NLS-1$
        }
    };
    /**
     * Used by <code>GraphExplorer.setTextAndImage</code>.
     */
    public static final QueryKey<Collection<ImageDecoratorFactory>> IMAGE_DECORATOR_FACTORIES = new QueryKey<Collection<ImageDecoratorFactory>>() {
        @Override
        public String toString() {
            return "IMAGE_DECORATOR_FACTORIES"; //$NON-NLS-1$
        }
    };
    /**
     * Used by <code>GraphExplorer.setTextAndImage</code>.
     */
    public static final QueryKey<Collection<ImageDecorator>> IMAGE_DECORATORS = new QueryKey<Collection<ImageDecorator>>() {
        @Override
        public String toString() {
            return "IMAGE_DECORATORS"; //$NON-NLS-1$
        }
    };

    public static final QueryKey<Labeler> SELECTED_LABELER = new QueryKey<Labeler>() {
        @Override
        public String toString() {
            return "SELECTED_LABELER"; //$NON-NLS-1$
        }
    };

    public static final QueryKey<Imager> SELECTED_IMAGER = new QueryKey<Imager>() {
        @Override
        public String toString() {
            return "SELECTED_IMAGER"; //$NON-NLS-1$
        }
    };

    public static final QueryKey<Collection<LabelDecoratorFactory>> SELECTED_LABEL_DECORATOR_FACTORIES = new QueryKey<Collection<LabelDecoratorFactory>>() {
        @Override
        public String toString() {
            return "SELECTED_LABEL_DECORATOR_FACTORIES"; //$NON-NLS-1$
        }
    };

    public static final QueryKey<Collection<ImageDecoratorFactory>> SELECTED_IMAGE_DECORATOR_FACTORIES = new QueryKey<Collection<ImageDecoratorFactory>>() {
        @Override
        public String toString() {
            return "SELECTED_IMAGE_DECORATOR_FACTORIES"; //$NON-NLS-1$
        }
    };

    /**
     * A query for all available labeler factories for the current input.
     * <p>
     * Depends on: {@link #INPUT}
     * <p>
     * Returns: a non-null collection
     */
    public static final QueryKey<Collection<LabelerFactory>> LABELER_FACTORIES = new QueryKey<Collection<LabelerFactory>>() {
        @Override
        public String toString() {
            return "LABELER_FACTORIES"; //$NON-NLS-1$
        }
    };

    /**
     * A query for all available imager factories for the current input.
     * <p>
     * Depends on: {@link #INPUT}
     * <p>
     * Returns: a non-null collection
     */
    public static final QueryKey<Collection<ImagerFactory>> IMAGER_FACTORIES = new QueryKey<Collection<ImagerFactory>>() {
        @Override
        public String toString() {
            return "IMAGER_FACTORIES"; //$NON-NLS-1$
        }
    };

    public static class LabelerKey extends ParametrizedPrimitiveQueryKey<Labeler> {
        public LabelerKey(LabelerFactory factory) {
            super(factory);
            assert factory != null;
        }
        @Override
        public String getKeyName() {
            return "LABELER"; //$NON-NLS-1$
        }
    };

    public static class CheckedStateKey extends ParametrizedPrimitiveQueryKey<CheckedState> {
        public CheckedStateKey(CheckedStateFactory factory) {
            super(factory);
            assert factory != null;
        }
        @Override
        public String getKeyName() {
            return "CHECK_STATE"; //$NON-NLS-1$
        }
    };

    public static class LabelDecoratorKey extends ParametrizedPrimitiveQueryKey<LabelDecorator> {
        public LabelDecoratorKey(LabelDecoratorFactory factory) {
            super(factory);
            assert factory != null;
        }
        @Override
        public String getKeyName() {
            return "LABEL_DECORATOR"; //$NON-NLS-1$
        }
    };

    public static class ImagerKey extends ParametrizedPrimitiveQueryKey<Imager> {
        public ImagerKey(ImagerFactory factory) {
            super(factory);
            assert factory != null;
        }
        @Override
        public String getKeyName() {
            return "IMAGER"; //$NON-NLS-1$
        }
    };

    public static class ImageDecoratorKey extends ParametrizedPrimitiveQueryKey<ImageDecorator> {
        public ImageDecoratorKey(ImageDecoratorFactory factory) {
            super(factory);
            assert factory != null;
        }
        @Override
        public String getKeyName() {
            return "IMAGE_DECORATOR"; //$NON-NLS-1$
        }
    };

    /**
     * A query that returns a collection of {@link SelectionRequest}s based on
     * the current input. The selection requests are used to dictate whether a
     * child INodeContext of the input INodeContext gets filtered out from the
     * result of the {@link #PRUNED_CHILDREN} query.
     * 
     * <p>
     * The query is free to return <code>null</code> if there are no filters for
     * the query input.
     * </p>
     */
    public static final PrimitiveQueryKey<Collection<SelectionRequest>> SELECTION_REQUESTS = new PrimitiveQueryKey<Collection<SelectionRequest>>() {
        @Override
        public String toString() {
            return "SELECTION_REQUEST"; //$NON-NLS-1$
        }
    };

    //------------------------------------------------------------------------

    /**
     * A query that requests for the currently selected viewpoint of a node
     * context, gets all of its children from the viewpoint and prunes the set
     * of children based on active filters (selection requests).
     * <p>
     * Depends on: {@link #SELECTED_VIEWPOINT}, {@link #SELECTION_REQUESTS}
     * <p>
     * Returns: A {@link PrunedChildrenResult} which contains the pruned set of
     * children as INodeContext[] and the original amount of children before
     * pruning them.
     */
    public static final QueryKey<PrunedChildrenResult> PRUNED_CHILDREN = new QueryKey<PrunedChildrenResult>() {
        @Override
        public String toString() {
            return "PRUNED_CHILDREN"; //$NON-NLS-1$
        }
    };

    /**
     * A query that requests for the currently selected viewpoint of a node
     * context and counts its children.
     * <p>
     * Depends on: {@link #VIEWPOINT_CONTRIBUTIONS}
     * <p>
     * Returns: <code>false</code> only if it is certain that there are no children
     */
    public static final QueryKey<Boolean> HAS_CHILDREN = new QueryKey<Boolean>() {
        @Override
        public String toString() {
            return "HAS_CHILDREN"; //$NON-NLS-1$
        }
    };

    /**
     * A query to transform INodeContext's into ComparableContexts that can be
     * sorted.
     * <p>
     * Depends on: {@link #PRUNED_CHILDREN}
     * <p>
     * Returns: ComparableContext[] of the same size as the input or
     * <code>null</code> if no selected ComparableFactory could be discovered
     * which indicates that child nodes cannot be sorted.
     */
    public static final QueryKey<ComparableContext[]> COMPARABLE_CHILDREN = new QueryKey<ComparableContext[]>() {
        @Override
        public String toString() {
            return "COMPARABLE_CHILDREN"; //$NON-NLS-1$
        }
    };

    /**
     * A query for all available comparable factories for the current input.
     * <p>
     * Depends on: {@link #INPUT}
     * <p>
     * Returns: a non-null collection
     */
    public static final QueryKey<Collection<ComparableContextFactory>> COMPARABLE_FACTORIES = new QueryKey<Collection<ComparableContextFactory>>() {
        @Override
        public String toString() {
            return "COMPARABLE_FACTORIES"; //$NON-NLS-1$
        }
    };

    public static class SelectedComparableFactoryKey extends ParametrizedPrimitiveQueryKey<ComparableContextFactory> {
        public SelectedComparableFactoryKey(Collection<ComparableContextFactory> factories) {
            super(factories);
            assert factories != null;
        }
        @Override
        public String getKeyName() {
            return "SELECTED_COMPARABLE_FACTORY"; //$NON-NLS-1$
        }
    }

    /**
     * Returns the final child nodes for a node after pruning and sorting them.
     * If you need to discover the set of children for a node you should use
     * {@link BuiltinKeys#PRUNED_CHILDREN} instead if you don't need the
     * sorting. To get the non-pruned set of all children of a node
     * <p>
     * Depends on: {@link BuiltinKeys#COMPARABLE_CHILDREN},
     * {@link BuiltinKeys#PRUNED_CHILDREN}
     * <p>
     * Returns: an array of {@link NodeContext}s.
     */
    public static final QueryKey<NodeContext[]> FINAL_CHILDREN = new QueryKey<NodeContext[]>() {
        @Override
        public String toString() {
            return "FINAL_CHILDREN"; //$NON-NLS-1$
        }
    };

    /**
     * Returns whether a node is expanded in the UI or not.
     * <p>
     * Returns: {@link Boolean#TRUE} if expanded, {@link Boolean#FALSE} if not.
     */
    public static final PrimitiveQueryKey<Boolean> IS_EXPANDED = new PrimitiveQueryKey<Boolean>() {
        @Override
        public String toString() {
            return "IS_EXPANDED"; //$NON-NLS-1$
        }
    };

    /**
     * Tells whether a node is "checked" in the UI or not. This only takes
     * effect if the GraphExplorer was created with support for checkedness
     * visualisation.
     * <p>
     * Returns: {@link CheckedState}
     */
    public static final QueryKey<CheckedState> IS_CHECKED = new QueryKey<CheckedState>() {
        @Override
        public String toString() {
            return "IS_CHECKED"; //$NON-NLS-1$
        }
    };

    /**
     * Returns how many children should maximally be shown for a node. By
     * default the set of shown children is pruned if there are too many of them
     * to be shown.
     * <p>
     * Returns:
     * <ul>
     * <li>0 < x &le; {@value Integer#MAX_VALUE} to tell the maximum amount of
     * children to show</li>
     * <li><code>null</code> indicates don't care</li>
     * </ul>
     * May not return < 0.
     */
    public static final PrimitiveQueryKey<Integer> SHOW_MAX_CHILDREN = new PrimitiveQueryKey<Integer>() {
        @Override
        public String toString() {
            return "SHOW_MAX_CHILDREN"; //$NON-NLS-1$
        }
    };

    public static class IsRootKey implements ConstantKey<Object> {
        private IsRootKey() {}
        @Override
        public String toString() {
            return "IS_ROOT"; //$NON-NLS-1$
        }
    };

    /**
     * Set to {@link Boolean#TRUE} for a NodeContext that describes the root
     * input of a {@link GraphExplorer}.
     */
    public static final ConstantKey<Object> IS_ROOT = new IsRootKey();

}
