package org.simantics.audit.client;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collections;
import java.util.Map;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.simantics.audit.Activator;
import org.simantics.audit.AuditLoggingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AuditLoggingAPIClient {

    private static final Logger LOGGER = LoggerFactory.getLogger(AuditLoggingAPIClient.class);
    
    private Client httpClient;
    private WebTarget base;
    private String uuid;

    public AuditLoggingAPIClient(String id, String serverAddress) throws AuditLoggingException {
        ClientConfig configuration = new ClientConfig();
        configuration.register(JacksonFeature.class);
        httpClient = ClientBuilder.newClient(configuration);
        if (!serverAddress.startsWith("http://") && !serverAddress.startsWith("https://")) {
            serverAddress = "http://" + serverAddress;
        }
        base = httpClient.target(serverAddress);
        
        // see if registered already
        uuid = possibleUUIDFromFile();
        if (uuid == null) {
            // register
            register(id);
        }
    }

    public String getUuid() {
        return uuid;
    }

    private void register(String id) throws AuditLoggingException {
        try {
            Response response = base.path("register").request(MediaType.APPLICATION_JSON_TYPE).post(Entity.json(Collections.singletonMap("id", id)));
            Map<String, String> payload = response.readEntity(Map.class);
            String possibleUUID = payload.get("uuid");
            if (possibleUUID != null && !possibleUUID.isEmpty()) {
                persistUUID(possibleUUID);
            } else {
                LOGGER.warn("Invalid response received from {} for register with response payload {}", base.getUri(), payload);
            }
        } catch (Exception e) {
            throw new AuditLoggingException(e);
        }
    }
    
    private static Path auditLoggingFile() {
        return Activator.getLogLocation().resolve(".auditlogging");
    }
    
    private static String possibleUUIDFromFile() {
        Path auditLoggingFile = auditLoggingFile();
        if (Files.exists(auditLoggingFile)) {
            try {
                String possibleUUID = new String(Files.readAllBytes(auditLoggingFile));
                if (possibleUUID != null && !possibleUUID.isEmpty()) {
                    return possibleUUID;
                } else {
                    LOGGER.warn(".auditlogging file exists but is somehow corrupted");
                }
            } catch (IOException e) {
                LOGGER.error("Could not read .auditlogging file and related information", e);
            }
        }
        return null;
    }
    
    private void persistUUID(String possibleUUID) throws IOException {
        Path auditLoggingFile = auditLoggingFile();
        Files.write(auditLoggingFile, possibleUUID.getBytes(StandardCharsets.UTF_8), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        uuid = possibleUUID;
    }

    public void log(Map<String, Object> message) throws AuditLoggingException {
        try {
            Response response = base.path(uuid).path("log").request(MediaType.APPLICATION_JSON_TYPE).post(Entity.json(message));
        } catch (Exception e) {
            throw new AuditLoggingException(e);
        }
    }
    
    public void error(Map<String, Object> message) throws AuditLoggingException {
        try {
            Response response = base.path(uuid).path("error").request(MediaType.APPLICATION_JSON_TYPE).post(Entity.json(message));
        } catch (Exception e) {
            throw new AuditLoggingException(e);
        }
    }

    public void trace(Map<String, Object> message) throws AuditLoggingException {
        try {
            Response response = base.path(uuid).path("trace").request(MediaType.APPLICATION_JSON_TYPE).post(Entity.json(message));
        } catch (Exception e) {
            throw new AuditLoggingException(e);
        }
    }

}
