/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.workbench.internal;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.util.Util;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.eclipse.ui.actions.ContributionItemFactory;
import org.eclipse.ui.actions.OpenInNewWindowAction;
import org.eclipse.ui.application.ActionBarAdvisor;
import org.eclipse.ui.application.IActionBarConfigurer;
import org.eclipse.ui.ide.IDEActionFactory;
import org.eclipse.ui.internal.Workbench;
import org.eclipse.ui.internal.WorkbenchPlugin;
import org.eclipse.ui.internal.cheatsheets.actions.CheatSheetCategoryBasedSelectionAction;
import org.eclipse.ui.internal.registry.ActionSetRegistry;
import org.eclipse.ui.internal.registry.IActionSetDescriptor;

/**
 * An action bar advisor is responsible for creating, adding, and disposing of
 * the actions added to a workbench window. Each window will be populated with
 * new actions.
 */
public class SimanticsWorkbenchActionBarAdvisor extends ActionBarAdvisor {

    //private final WorkbenchActionBuilder ideAdvisor;

    private final IWorkbenchWindow window;

    /**
     * Indicates if the action builder has been disposed
     */
    private boolean                isDisposed = false;

    // Actions - important to allocate these only in makeActions, and then use
    // them
    // in the fill methods. This ensures that the actions aren't recreated
    // when fillActionBars is called with FILL_PROXY.

    // File
    private IWorkbenchAction       closeAction;

    private IWorkbenchAction       closeAllAction;

    private IWorkbenchAction       closeOthersAction;

    private IWorkbenchAction       closeAllSavedAction;

//    private IWorkbenchAction       saveAction;

//    protected IWorkbenchAction       printAction;

    protected IWorkbenchAction       openWorkspaceAction;

    private IWorkbenchAction       importResourcesAction;

    private IWorkbenchAction       exportResourcesAction;

    protected IWorkbenchAction       quitAction;

    // Edit
    protected IWorkbenchAction       undoAction;

    protected IWorkbenchAction       redoAction;

    protected IWorkbenchAction       cutAction;

    protected IWorkbenchAction       copyAction;

    protected IWorkbenchAction       pasteAction;

    protected IWorkbenchAction       deleteAction;

    protected IWorkbenchAction       selectAllAction;

    // Navigate
    private IWorkbenchAction       backwardHistoryAction;

    private IWorkbenchAction       forwardHistoryAction;

    private IWorkbenchAction       goIntoAction;

    private IWorkbenchAction       backAction;

    private IWorkbenchAction       forwardAction;

    private IWorkbenchAction       upAction;

    private IWorkbenchAction       nextAction;

    private IWorkbenchAction       previousAction;

    // Window
    private IWorkbenchAction       newWindowAction;

    private IWorkbenchAction       newEditorAction;

    private IWorkbenchAction       savePerspectiveAction;

    private IWorkbenchAction       resetPerspectiveAction;

    private IWorkbenchAction       editActionSetAction;

    private IWorkbenchAction       closePerspAction;

    private IWorkbenchAction       closeAllPerspsAction;

    private IWorkbenchAction       openPreferencesAction;

    // Window/Navigation
    private IWorkbenchAction       showViewMenuAction;

    private IWorkbenchAction       showPartPaneMenuAction;

    private IWorkbenchAction       maximizePartAction;

    private IWorkbenchAction       minimizePartAction;

    private IWorkbenchAction       nextEditorAction;

    private IWorkbenchAction       prevEditorAction;

    private IWorkbenchAction       activateEditorAction;

    private IWorkbenchAction       switchToEditorAction;

    private IWorkbenchAction       openEditorDropDownAction;

    private IWorkbenchAction       nextPartAction;

    private IWorkbenchAction       prevPartAction;

    private IWorkbenchAction       nextPerspectiveAction;

    private IWorkbenchAction       prevPerspectiveAction;

    // Help
    private IWorkbenchAction       introAction;

    private IWorkbenchAction       tocAction;

    private IWorkbenchAction       searchAction;

    private IWorkbenchAction       dynamicHelpAction;

    private IAction                cheatSheetsAction;

    private IWorkbenchAction       aboutAction;


    /**
     * Constructs a new action builder which contributes actions
     * to the given window.
     * 
     * @param configurer the action bar configurer for the window
     */
    public SimanticsWorkbenchActionBarAdvisor(IActionBarConfigurer configurer) {
        super(configurer);
        //ideAdvisor = new WorkbenchActionBuilder(configurer);
        window = configurer.getWindowConfigurer().getWindow();
    }

    /**
     * Returns the window to which this action builder is contributing.
     */
    private IWorkbenchWindow getWindow() {
        return window;
    }

    /**
     * Disposes any resources and unhooks any listeners that are no longer needed.
     * Called when the window is closed.
     */
    @Override
    public void dispose() {
        if (isDisposed) {
            return;
        }

        isDisposed = true;

        // null out actions to make leak debugging easier
        // File
        closeAction = null;
        closeAllAction = null;
        closeAllSavedAction = null;
        closeOthersAction = null;
//        saveAction = null;
//        printAction = null;
        openWorkspaceAction = null;
        importResourcesAction = null;
        exportResourcesAction = null;
        quitAction = null;

        // Edit
        undoAction = null;
        redoAction = null;
        cutAction = null;
        copyAction = null;
        pasteAction = null;
        deleteAction = null;
        selectAllAction = null;

        // Navigate
        backwardHistoryAction = null;
        forwardHistoryAction = null;
        goIntoAction = null;
        backAction = null;
        forwardAction = null;
        upAction = null;
        nextAction = null;
        previousAction = null;

        // Window
        newWindowAction = null;
        newEditorAction = null;
        openPreferencesAction = null;
        savePerspectiveAction = null;
        resetPerspectiveAction = null;
        editActionSetAction = null;
        closePerspAction = null;
        closeAllPerspsAction = null;
        openPreferencesAction = null;

        // Window/Navigation
        showViewMenuAction = null;
        showPartPaneMenuAction = null;
        maximizePartAction = null;
        minimizePartAction = null;
        nextPartAction = null;
        prevPartAction = null;
        nextPerspectiveAction = null;
        prevPerspectiveAction = null;

        nextEditorAction = null;
        prevEditorAction = null;
        activateEditorAction = null;
        switchToEditorAction = null;
        openEditorDropDownAction = null;

        // Help
        introAction = null;
        tocAction = null;
        searchAction = null;
        dynamicHelpAction = null;
        cheatSheetsAction = null;
        aboutAction = null;

        super.dispose();
    }

    @Override
    protected void makeActions(final IWorkbenchWindow window) {
        // Creates the actions and registers them.
        // Registering is needed to ensure that key bindings work.
        // The corresponding commands keybindings are defined in the plugin.xml
        // file.
        // Registering also provides automatic disposal of the actions when
        // the window is closed.

        // File
        closeAction = ActionFactory.CLOSE.create(window);
        register(closeAction);
        closeAllAction = ActionFactory.CLOSE_ALL.create(window);
        register(closeAllAction);
        closeOthersAction = ActionFactory.CLOSE_OTHERS.create(window);
        register(closeOthersAction);
        closeAllSavedAction = ActionFactory.CLOSE_ALL_SAVED.create(window);
        register(closeAllSavedAction);
//        saveAction = ActionFactory.SAVE.create(window);
//        register(saveAction);
//        printAction = ActionFactory.PRINT.create(window);
//        register(printAction);
        openWorkspaceAction = IDEActionFactory.OPEN_WORKSPACE.create(window);
        register(openWorkspaceAction);
        importResourcesAction = ActionFactory.IMPORT.create(window);
        register(importResourcesAction);
        exportResourcesAction = ActionFactory.EXPORT.create(window);
        register(exportResourcesAction);
        quitAction = ActionFactory.QUIT.create(window);
        register(quitAction);

        // Edit
//        undoAction = ActionFactory.UNDO.create(window);
//        register(undoAction);
//        redoAction = ActionFactory.REDO.create(window);
//        register(redoAction);
//        cutAction = ActionFactory.CUT.create(window);
//        register(cutAction);
//        copyAction = ActionFactory.COPY.create(window);
//        register(copyAction);
//        pasteAction = ActionFactory.PASTE.create(window);
//        register(pasteAction);
//        deleteAction = ActionFactory.DELETE.create(window);
//        register(deleteAction);
//        selectAllAction = ActionFactory.SELECT_ALL.create(window);
//        register(selectAllAction);

        // Navigate
        forwardHistoryAction = ActionFactory.FORWARD_HISTORY.create(window);
        register(forwardHistoryAction);
        backwardHistoryAction = ActionFactory.BACKWARD_HISTORY.create(window);
        register(backwardHistoryAction);
        goIntoAction = ActionFactory.GO_INTO.create(window);
        register(goIntoAction);
        backAction = ActionFactory.BACK.create(window);
        register(backAction);
        forwardAction = ActionFactory.FORWARD.create(window);
        register(forwardAction);
        upAction = ActionFactory.UP.create(window);
        register(upAction);
        nextAction = ActionFactory.NEXT.create(window);
        //nextAction.setImageDescriptor(IDEInternalWorkbenchImages.getImageDescriptor(IDEInternalWorkbenchImages.IMG_ETOOL_NEXT_NAV));
        register(nextAction);
        previousAction = ActionFactory.PREVIOUS.create(window);
        //previousAction.setImageDescriptor(IDEInternalWorkbenchImages.getImageDescriptor(IDEInternalWorkbenchImages.IMG_ETOOL_PREVIOUS_NAV));
        register(previousAction);


        // Window
        newWindowAction = OPEN_NEW_WINDOW.create(getWindow());
        newWindowAction.setText("&New Window");
        register(newWindowAction);
        newEditorAction = ActionFactory.NEW_EDITOR.create(window);
        register(newEditorAction);
        editActionSetAction = ActionFactory.EDIT_ACTION_SETS.create(window);
        register(editActionSetAction);
        savePerspectiveAction = ActionFactory.SAVE_PERSPECTIVE.create(window);
        register(savePerspectiveAction);
        resetPerspectiveAction = ActionFactory.RESET_PERSPECTIVE.create(window);
        register(resetPerspectiveAction);
        closePerspAction = ActionFactory.CLOSE_PERSPECTIVE.create(window);
        register(closePerspAction);
        closeAllPerspsAction = ActionFactory.CLOSE_ALL_PERSPECTIVES.create(window);
        register(closeAllPerspsAction);
        openPreferencesAction = ActionFactory.PREFERENCES.create(window);
//        preferencesAction = new OpenPreferencesAction(window);
//        preferencesAction.setId("preferences");
        register(openPreferencesAction);

        // Window/Navigation: Actions for invisible accelerators
        showViewMenuAction = ActionFactory.SHOW_VIEW_MENU.create(window);
        register(showViewMenuAction);
        showPartPaneMenuAction = ActionFactory.SHOW_PART_PANE_MENU.create(window);
        register(showPartPaneMenuAction);
        maximizePartAction = ActionFactory.MAXIMIZE.create(window);
        register(maximizePartAction);
        minimizePartAction = ActionFactory.MINIMIZE.create(window);
        register(minimizePartAction);

        nextEditorAction = ActionFactory.NEXT_EDITOR.create(window);
        register(nextEditorAction);
        prevEditorAction = ActionFactory.PREVIOUS_EDITOR.create(window);
        register(prevEditorAction);
        ActionFactory.linkCycleActionPair(nextEditorAction, prevEditorAction);
        activateEditorAction = ActionFactory.ACTIVATE_EDITOR.create(window);
        register(activateEditorAction);
        switchToEditorAction = ActionFactory.SHOW_OPEN_EDITORS.create(window);
        register(switchToEditorAction);
        openEditorDropDownAction = ActionFactory.SHOW_WORKBOOK_EDITORS.create(window);
        register(openEditorDropDownAction);

        nextPartAction = ActionFactory.NEXT_PART.create(window);
        register(nextPartAction);
        prevPartAction = ActionFactory.PREVIOUS_PART.create(window);
        register(prevPartAction);
        ActionFactory.linkCycleActionPair(nextPartAction, prevPartAction);
        nextPerspectiveAction = ActionFactory.NEXT_PERSPECTIVE.create(window);
        register(nextPerspectiveAction);
        prevPerspectiveAction = ActionFactory.PREVIOUS_PERSPECTIVE.create(window);
        register(prevPerspectiveAction);
        ActionFactory.linkCycleActionPair(nextPerspectiveAction, prevPerspectiveAction);

        // Help
        if (window.getWorkbench().getIntroManager().hasIntro()) {
            introAction = ActionFactory.INTRO.create(window);
            register(introAction);
        }
        tocAction = ActionFactory.HELP_CONTENTS.create(window);
        register(tocAction);
        searchAction = ActionFactory.HELP_SEARCH.create(window);
        register(searchAction);
        dynamicHelpAction = ActionFactory.DYNAMIC_HELP.create(window);
        register(dynamicHelpAction);
        cheatSheetsAction = new CheatSheetCategoryBasedSelectionAction("&Cheat Sheets...");
        cheatSheetsAction.setId("org.eclipse.ui.help.cheatSheetsAction");
        register(cheatSheetsAction);
        aboutAction = ActionFactory.ABOUT.create(window);
        register(aboutAction);
    }

    /*
     * TODO: HACK for removing Eclipse's own forced and totally unwanted actionSet extensions.
     * Remove if a better way to achieve the same effect is found.
     */
    protected void hackRemoveUnwantedNavigationActions() {
        ActionSetRegistry reg = WorkbenchPlugin.getDefault().getActionSetRegistry();
        IActionSetDescriptor[] actionSets = reg.getActionSets();
        String actionSetId[] = {
                "org.eclipse.ui.edit.text.actionSet.navigation",
                "org.eclipse.ui.edit.text.actionSet.annotationNavigation",
                "org.eclipse.ui.NavigateActionSet"
        };
        Set<String> actionSetIds = new HashSet<String>();
        for (String s : actionSetId)
            actionSetIds.add(s);

        for (int i = 0; i < actionSets.length; i++) {
            if ((!actionSetIds.contains(actionSets[i].getId())))
                continue;
            IExtension ext = actionSets[i].getConfigurationElement().getDeclaringExtension();
            reg.removeExtension(ext, new Object[] { actionSets[i] });
        }
    }

    @Override
    protected void fillMenuBar(IMenuManager menuBar) {
        hackRemoveUnwantedNavigationActions();

        menuBar.add(createFileMenu());
        menuBar.add(createEditMenu());
        menuBar.add(createNavigateMenu());
        menuBar.add(createProjectMenu());
        menuBar.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
        menuBar.add(createWindowMenu());
        menuBar.add(createHelpMenu());
    }

    @Override
    protected void fillStatusLine(IStatusLineManager statusLine) {
        //System.out.println("fillStatusLine: " + statusLine);
        //statusLine.appendToGroup(StatusLineManager.BEGIN_GROUP, GlobalStatusLineContributionItem.getInstance());
//        statusLine.add(new GraphRequestStatusContribution(window));
//        statusLine.add(new DumpHeapContribution(window));
//        statusLine.add(new DumpStackTracesContribution(window));
//        statusLine.add(new FlushRequestsContribution(window));
//        statusLine.add(new SearchContribution(window));
    }

    protected IContributionItem createFileMenu() {
        //MenuManager menu = new MenuManager("&File", IWorkbenchActionConstants.M_FILE);
        MenuManager menu = new MenuManager("&File", ISimanticsWorkbenchConstants.M_FILE);

        menu.add(new GroupMarker(IWorkbenchActionConstants.FILE_START));
//        {
//            // create the New submenu, using the same id for it as the New action
//            String newText = IDEWorkbenchMessages.Workbench_new;
//            String newId = ActionFactory.NEW.getId();
//            MenuManager newMenu = new MenuManager(newText, newId) {
//                public String getMenuText() {
//                    String result = super.getMenuText();
//                    if (newQuickMenu == null) {
//                        return result;
//                    }
//                    String shortCut = newQuickMenu.getShortCutString();
//                    if (shortCut == null) {
//                        return result;
//                    }
//                    return result + "\t" + shortCut; //$NON-NLS-1$
//                }
//            };
//            newMenu.add(new Separator(newId));
//            this.newWizardMenu = new NewWizardMenu(getWindow());
//            newMenu.add(this.newWizardMenu);
//            newMenu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
//            menu.add(newMenu);
//        }

        menu.add(new GroupMarker(IWorkbenchActionConstants.NEW_EXT));
        menu.add(new Separator());

        menu.add(closeAction);
        menu.add(closeAllAction);
        //menu.add(closeAllSavedAction);
        menu.add(new GroupMarker(IWorkbenchActionConstants.CLOSE_EXT));
        menu.add(new Separator());
//        menu.add(saveAction);

        menu.add(new GroupMarker(IWorkbenchActionConstants.SAVE_EXT));
        menu.add(new Separator());
//        menu.add(printAction);
        menu.add(new GroupMarker(IWorkbenchActionConstants.PRINT_EXT));
        menu.add(new Separator());
        menu.add(openWorkspaceAction);
        menu.add(new GroupMarker(IWorkbenchActionConstants.OPEN_EXT));
        menu.add(new Separator());
//        menu.add(importResourcesAction);
//        menu.add(exportResourcesAction);
        menu.add(new GroupMarker(IWorkbenchActionConstants.IMPORT_EXT));
        menu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));

        menu.add(new Separator());

        // If we're on OS X we shouldn't show this command in the File menu. It
        // should be invisible to the user. However, we should not remove it -
        // the carbon UI code will do a search through our menu structure
        // looking for it when Cmd-Q is invoked (or Quit is chosen from the
        // application menu.
        ActionContributionItem quitItem = new ActionContributionItem(quitAction);
        quitItem.setVisible(!Util.isMac());
        menu.add(quitItem);
        menu.add(new GroupMarker(IWorkbenchActionConstants.FILE_END));

        return menu;
    }

    protected IContributionItem createEditMenu() {
        MenuManager menu = new MenuManager("&Edit", IWorkbenchActionConstants.M_EDIT);

//        menu.add(undoAction);
//        menu.add(redoAction);
//        menu.add(new Separator());
//        menu.add(cutAction);
//        menu.add(copyAction);
//        menu.add(pasteAction);
//        menu.add(new Separator());
//        menu.add(deleteAction);
//        menu.add(selectAllAction);
        menu.add(new GroupMarker(IWorkbenchActionConstants.EDIT_START));
//        menu.add(new GroupMarker(IWorkbenchActionConstants.UNDO_EXT));
//        menu.add(new GroupMarker(IWorkbenchActionConstants.CUT_EXT));
        menu.add(new GroupMarker(IWorkbenchActionConstants.FIND_EXT));
        menu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
        menu.add(new GroupMarker(IWorkbenchActionConstants.EDIT_END));

        return menu;
    }

    /**
     * Creates and returns the Navigate menu.
     */
    protected MenuManager createNavigateMenu() {
        MenuManager menu = new MenuManager("&Navigate", IWorkbenchActionConstants.M_NAVIGATE);
        menu.add(new GroupMarker(IWorkbenchActionConstants.NAV_START));
//        menu.add(goIntoAction);

//        MenuManager goToSubMenu = new MenuManager("&Go To", IWorkbenchActionConstants.GO_TO);
//        menu.add(goToSubMenu);
//        goToSubMenu.add(backAction);
//        goToSubMenu.add(forwardAction);
//        goToSubMenu.add(upAction);
//        goToSubMenu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));

        menu.add(new Separator(IWorkbenchActionConstants.OPEN_EXT));
        for (int i = 2; i < 5; ++i) {
            menu.add(new Separator(IWorkbenchActionConstants.OPEN_EXT + i));
        }
        menu.add(new Separator(IWorkbenchActionConstants.SHOW_EXT));
        {

//            MenuManager showInSubMenu = new MenuManager("Sho&w In", "showIn") {
//                public String getMenuText() {
//                    String result = super.getMenuText();
//                    if (showInQuickMenu == null) {
//                        return null;
//                    }
//                    String shortCut = showInQuickMenu.getShortCutString();
//                    if (shortCut == null) {
//                        return result;
//                    }
//                    return result + "\t" + shortCut; //$NON-NLS-1$
//                }
//            };
//            showInSubMenu.add(ContributionItemFactory.VIEWS_SHOW_IN
//                    .create(getWindow()));
//            menu.add(showInSubMenu);
        }
        for (int i = 2; i < 5; ++i) {
            menu.add(new Separator(IWorkbenchActionConstants.SHOW_EXT + i));
        }
        menu.add(new Separator());
//        menu.add(nextAction);
//        menu.add(previousAction);
        menu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
        menu.add(new GroupMarker(IWorkbenchActionConstants.NAV_END));

        //TBD: Location of this actions
        menu.add(new Separator());
//        menu.add(backwardHistoryAction);
//        menu.add(forwardHistoryAction);
        return menu;
    }

    /**
     * Creates and returns the Project menu.
     */
    protected MenuManager createProjectMenu() {
        MenuManager menu = new MenuManager("&Project", ISimanticsWorkbenchConstants.M_PROJECT);
        menu.add(new Separator(IWorkbenchActionConstants.PROJ_START));

//        menu.add(openProjectAction);
//        menu.add(closeProjectAction);
//        menu.add(new GroupMarker(IWorkbenchActionConstants.OPEN_EXT));
//        menu.add(new Separator());
//        menu.add(buildAllAction);
//        menu.add(buildProjectAction);
//        addWorkingSetBuildActions(menu);
//        menu.add(cleanAction);
//        menu.add(toggleAutoBuildAction);
//        menu.add(new GroupMarker(IWorkbenchActionConstants.BUILD_EXT));
//        menu.add(new Separator());

        menu.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
        menu.add(new GroupMarker(IWorkbenchActionConstants.PROJ_END));
        menu.add(new Separator());
//        menu.add(projectPropertyDialogAction);
        return menu;
    }


    protected IContributionItem createWindowMenu() {
        MenuManager menu = new MenuManager("&Window", IWorkbenchActionConstants.M_WINDOW);

        menu.add(newWindowAction);
//        menu.add(newEditorAction);

        menu.add(new Separator());
        addPerspectiveActions(menu);
        menu.add(new Separator());
        addKeyboardShortcuts(menu);
        Separator sep = new Separator(IWorkbenchActionConstants.MB_ADDITIONS);
        sep.setVisible(!Util.isMac());
        menu.add(sep);
        sep = new Separator(IWorkbenchActionConstants.MB_ADDITIONS + ".end"); //$NON-NLS-1$
        sep.setVisible(!Util.isMac());
        menu.add(sep);

        // See the comment for quit in createFileMenu
        ActionContributionItem openPreferencesItem = new ActionContributionItem(openPreferencesAction);
        openPreferencesItem.setVisible(!Util.isMac());
        menu.add(openPreferencesItem);

        menu.add(ContributionItemFactory.OPEN_WINDOWS.create(getWindow()));
        return menu;
    }

    protected IContributionItem createHelpMenu() {
        // Help
        MenuManager menu = new MenuManager("&Help", IWorkbenchActionConstants.M_HELP);

        menu.add(new GroupMarker(IWorkbenchActionConstants.HELP_START));
        if (introAction != null)
            menu.add(introAction);
        menu.add(new GroupMarker("group.intro.ext")); //$NON-NLS-1$
        menu.add(new Separator("group.main"));
        //addSeparatorOrGroupMarker(menu, "group.main"); //$NON-NLS-1$
        menu.add(tocAction);
        menu.add(searchAction);
        menu.add(dynamicHelpAction);
        menu.add(new GroupMarker("group.intro.ext")); //$NON-NLS-1$
        menu.add(new Separator("group.main")); //$NON-NLS-1$
        //addSeparatorOrGroupMarker(menu, "group.main"); //$NON-NLS-1$
        addSeparatorOrGroupMarker(menu, "group.assist"); //$NON-NLS-1$
        menu.add(new GroupMarker("group.main.ext")); //$NON-NLS-1$
        menu.add(cheatSheetsAction);
        addSeparatorOrGroupMarker(menu, "group.tutorials"); //$NON-NLS-1$
        addSeparatorOrGroupMarker(menu, "group.tools"); //$NON-NLS-1$
        addSeparatorOrGroupMarker(menu, "group.updates"); //$NON-NLS-1$
        menu.add(new GroupMarker(IWorkbenchActionConstants.HELP_END));
        //addSeparatorOrGroupMarker(menu, IWorkbenchActionConstants.MB_ADDITIONS);
        menu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
        //menu.add(new Separator());
        menu.add(new Separator("group.about")); //$NON-NLS-1$

        ActionContributionItem aboutItem = new ActionContributionItem(aboutAction);
        aboutItem.setVisible(!Util.isMac());
        menu.add(aboutItem);
        menu.add(new GroupMarker("group.about.ext")); //$NON-NLS-1$
        return menu;
    }

    /**
     * Adds the perspective actions to the specified menu.
     */
    private void addPerspectiveActions(MenuManager menu) {
        {
            MenuManager changePerspMenuMgr = new MenuManager("&Open Perspective", "openPerspective");
            IContributionItem changePerspMenuItem = ContributionItemFactory.PERSPECTIVES_SHORTLIST.create(getWindow());
            changePerspMenuMgr.add(changePerspMenuItem);
            menu.add(changePerspMenuMgr);
        }
        {
            MenuManager showViewMenuMgr = new MenuManager("Show &View", "showView");
            IContributionItem showViewMenu = ContributionItemFactory.VIEWS_SHORTLIST.create(getWindow());
            showViewMenuMgr.add(showViewMenu);
            menu.add(showViewMenuMgr);
        }
        menu.add(new Separator());
        menu.add(editActionSetAction);
        menu.add(savePerspectiveAction);
        menu.add(resetPerspectiveAction);
        menu.add(closePerspAction);
        menu.add(closeAllPerspsAction);
    }

    /**
     * Adds the keyboard navigation submenu to the specified menu.
     */
    private void addKeyboardShortcuts(MenuManager menu) {
        MenuManager subMenu = new MenuManager("Navi&gation", "shortcuts");
        menu.add(subMenu);

        subMenu.add(showPartPaneMenuAction);
        subMenu.add(showViewMenuAction);
        subMenu.add(new Separator());
        subMenu.add(maximizePartAction);
        subMenu.add(minimizePartAction);
        subMenu.add(new Separator());
        subMenu.add(activateEditorAction);
        subMenu.add(nextEditorAction);
        subMenu.add(prevEditorAction);
        subMenu.add(switchToEditorAction);
        subMenu.add(openEditorDropDownAction);
        subMenu.add(new Separator());
        subMenu.add(nextPartAction);
        subMenu.add(prevPartAction);
        subMenu.add(new Separator());
        subMenu.add(nextPerspectiveAction);
        subMenu.add(prevPerspectiveAction);
    }

    /*
    private static String[] getPreferencePageIDs() {
        PreferenceManager mgr = PlatformUI.getWorkbench().getPreferenceManager();
        List nodes = mgr.getElements(PreferenceManager.POST_ORDER);
        String result[] = new String[nodes.size()];
        int i=0;
        for (Object o : nodes) {
            IPreferenceNode node = (IPreferenceNode) o;
            result[i++] = node.getId();
        }
        return result;
    }
     */

    /*
    class OpenPreferencesAction extends Action implements ActionFactory.IWorkbenchAction {
        IWorkbenchWindow workbenchWindow;
        public OpenPreferencesAction(IWorkbenchWindow window) {
            super(WorkbenchMessages.OpenPreferences_text);
            workbenchWindow = window;
            setActionDefinitionId("org.eclipse.ui.window.preferences");
            setToolTipText(WorkbenchMessages.OpenPreferences_toolTip);
            window.getWorkbench().getHelpSystem().setHelp(this,
                    IWorkbenchHelpContextIds.OPEN_PREFERENCES_ACTION);
        }
        public String[] getIDsToBeDisplayed() {
            String nodes[] = ApplicationActionBarAdvisor.getPreferencePageIDs();
            List<String> ids = new ArrayList<String>();
            for (String node: nodes) {
                if (node.startsWith("fi"))
                    ids.add(node);
            }
            return ids.toArray(new String[0]);
        }
        public void run() {
            if (workbenchWindow == null)
                return;

            String toBeDisplayed[] = getIDsToBeDisplayed();
            PreferenceDialog dialog =
                PreferencesUtil.createPreferenceDialogOn(null, null, toBeDisplayed, null);
            dialog.open();
        }
        public void dispose() {
            workbenchWindow = null;
        }
    }
     */

    /**
     * Our own version of the "New Window" in
     * <code>IWorkbenchAction.OPEN_NEW_WINDOW</code> action which allows
     * setting the page input IAdaptable.
     */
    public static final ActionFactory OPEN_NEW_WINDOW = new ActionFactory("openNewWindow") {//$NON-NLS-1$
        @Override
        public IWorkbenchAction create(IWorkbenchWindow window) {
            if (window == null) {
                throw new IllegalArgumentException();
            }
            IWorkbenchAction action = new NewWindowAction(window);
            action.setId(getId());
            return action;
        }
    };

    /**
     * This version of {@link OpenInNewWindowAction} will simply add a new
     * unique IAdaptable page input object for each window created through it.
     */
    static class NewWindowAction extends OpenInNewWindowAction {
        private IWorkbenchWindow workbenchWindow;
        public NewWindowAction(IWorkbenchWindow window) {
            super(window);
            this.workbenchWindow = window;
        }
        @Override
        public void dispose() {
            workbenchWindow = null;
            super.dispose();
        }
        @Override
        public void run() {
            IAdaptable defaultInput = ((Workbench) workbenchWindow.getWorkbench()).getDefaultPageInput();
            setPageInput(new DelegateAdaptable(defaultInput));
            super.run();
        }
    }

    static class DelegateAdaptable implements IAdaptable {
        IAdaptable target;
        public DelegateAdaptable(IAdaptable target) {
            this.target = target;
        }
        @SuppressWarnings({ "rawtypes" })
        @Override
        public Object getAdapter(Class adapter) {
            return (target != null) ? target.getAdapter(adapter) : null;
        }
    }

    /**
     * Adds a <code>GroupMarker</code> or <code>Separator</code> to a menu.
     * The test for whether a separator should be added is done by checking for
     * the existence of a preference matching the string
     * useSeparator.MENUID.GROUPID that is set to <code>true</code>.
     * 
     * @param menu
     *            the menu to add to
     * @param groupId
     *            the group id for the added separator or group marker
     */
    private void addSeparatorOrGroupMarker(MenuManager menu, String groupId) {
        String prefId = "useSeparator." + menu.getId() + "." + groupId; //$NON-NLS-1$ //$NON-NLS-2$
        boolean addExtraSeparators = Activator.getDefault().getPreferenceStore().getBoolean(prefId);
        if (addExtraSeparators) {
            menu.add(new Separator(groupId));
        } else {
            menu.add(new GroupMarker(groupId));
        }
    }

}
