/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * 23.8.2006
 */
package org.simantics.utils.ui.jface;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.jface.viewers.IPostSelectionProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.simantics.utils.ObjectUtils;

/**
 * BaseSelectionProvider is a base implementation ISelectionProvider -interface.
 * <p>
 * Usage:
 *  1) instantiate
 *  2) call {@link #setAndFireSelection(ISelection)} to send a selection event
 * 
 * <p>
 * Contains an empty StructuredSelection by default.
 * </p>
 * 
 * @author Toni Kalajainen
 */
public class BasePostSelectionProvider implements IPostSelectionProvider {

    protected ListenerList selectionListeners = new ListenerList();

    protected ListenerList postSelectionListeners = new ListenerList();

    protected ISelection selection = StructuredSelection.EMPTY;

    public void addSelectionChangedListener(ISelectionChangedListener listener) {
        selectionListeners.add(listener);
    }

    public ISelection getSelection() {
        return selection;
    }

    public void clearListeners() {
        selectionListeners.clear();
        postSelectionListeners.clear();
    }

    public void clearSelectionChangedListeners() {
        postSelectionListeners.clear();
    }

    public void clearPostSelectionChangedListeners() {
        postSelectionListeners.clear();
    }

    public void removeSelectionChangedListener(ISelectionChangedListener listener) {
        selectionListeners.remove(listener);
    }

    @Override
    public void addPostSelectionChangedListener(ISelectionChangedListener listener) {
        postSelectionListeners.add(listener);
    }

    @Override
    public void removePostSelectionChangedListener(ISelectionChangedListener listener) {
        postSelectionListeners.remove(listener);
    }

    public void setSelection(ISelection selection) {
        setSelectionWithoutFiring(selection);
    }

    protected Object[] getListeners() {
        return selectionListeners.getListeners();
    }

    protected Object[] getPostListeners() {
        return postSelectionListeners.getListeners();
    }

    /**
     * Notify other UIs that selection has changed
     * @param selection new selection
     */
    public void fireSelection(ISelection selection) {
        if (selection == null)
            return;
        SelectionChangedEvent e = new SelectionChangedEvent(this, selection);
        for (Object l : getListeners())
            ((ISelectionChangedListener) l).selectionChanged(e);
    }

    /**
     * Notify other UIs that selection has changed
     * @param selection new selection
     */
    public void firePostSelection(ISelection selection) {
        if (selection == null)
            return;
        final SelectionChangedEvent e = new SelectionChangedEvent(this, selection);
        for (Object o : postSelectionListeners.getListeners()) {
            final ISelectionChangedListener l = (ISelectionChangedListener) o;
            SafeRunner.run(new SafeRunnable() {
                @Override
                public void run() throws Exception {
                    l.selectionChanged(e);
                }
            });
        }
    }

    public void setSelectionWithoutFiring(ISelection selection) {
        this.selection = selection;
    }


    /**
     * Sets a new selection and always fires a SelectionChangedEvent about it.
     * 
     * @param selection the new selection
     */
    public void setAndFireSelection(ISelection selection) {
        setSelection(selection);
        fireSelection(selection);
    }

    /**
     * Sets the new selection for this provider and fires all selection change
     * listeners if the specified selection differs from the current selection.
     * If the selection is either the same object or considered equal to the
     * current selection, the listeners are not fired.
     * 
     * @param selection the new selection
     */
    public void setAndFireNonEqualSelection(ISelection selection) {
        ISelection old = getSelection();
        if (ObjectUtils.objectEquals(old, selection))
            return;

        this.selection = selection;
        if (selection != null && !selection.equals(old))
            fireSelection(selection);
    }

    public boolean selectionEquals(ISelection s) {
        if (s == selection)
            return true;
        if (s == null)
            // Old selection had to be non-null
            return true;
        return s.equals(selection);
    }

}
